/*****************************************************************************
 *
 *              RadioDriver.c -- user main program
 *
 *****************************************************************************
 * FileName:       	RadioDriver.c
 * Dependencies:	MSPI.h, Console.h, MRF24J40.h, RadioDriverDefs.h, stdio.h
 * Processor:       PIC18F/PIC24F
 * Compiler:        C18 02.20.00 or higher, C30 3.10 or higher
 *                  HITECH PICC-18 V8.10PL1 or higher
 * Linker:          MPLINK 03.40.00 or higher
 * Company:         Microchip Technology Incorporated
 *
 * Software License Agreement
 *
 * The software supplied herewith by Microchip Technology Incorporated
 * (the "Company") is intended and supplied to you, the Company's
 * customer, for use solely and exclusively with products manufactured
 * by the Company.
 *
 * The software is owned by the Company and/or its supplier, and is
 * protected under applicable copyright laws. All rights are reserved.
 * Any use in violation of the foregoing restrictions may subject the
 * user to criminal sanctions under applicable laws, as well as to
 * civil liability for the breach of the terms and conditions of this
 * license.
 *
 * THIS SOFTWARE IS PROVIDED IN AN "AS IS" CONDITION. NO WARRANTIES,
 * WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED
 * TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE COMPANY SHALL NOT,
 * IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL OR
 * CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
 *****************************************************************************/
#include "MSPI.h"
#include "Console.h"
#include "MRF24J40.h"
#include "RadioDriverDefs.h"
#include <stdio.h>
#if defined(__18F4620) || defined(__18F87J11)
    #include <delays.h>
    #include <p18cxxx.h>
#endif
#include <string.h>
							//include all the header files

//Define the configuration bits for each processor

    #if defined(__18F4620)
			//define the configuration bits for PICDEM Z board - PIC18F4620
        #pragma romdata CONFIG1H = 0x300001
        const rom unsigned char config1H = 0b00000110;      // HSPLL oscillator
    
        #pragma romdata CONFIG2L = 0x300002
        const rom unsigned char config2L = 0b00011111;      // Brown-out Reset Enabled in hardware @ 2.0V, PWRTEN disabled
    
        #pragma romdata CONFIG2H = 0x300003
        const rom unsigned char config2H = 0b00001010;      // HW WD disabled, 1:32 prescaler
    
        #pragma romdata CONFIG3H = 0x300005
        const rom unsigned char config3H = 0b10000000;      // PORTB digital on RESET
    
        #pragma romdata CONFIG4L = 0x300006
        const rom unsigned char config4L = 0b10000001;      // DEBUG disabled,
                                                            // XINST disabled
                                                            // LVP disabled
                                                            // STVREN enabled
		
	//define the configuration bits for PIC18 Explorer - PIC18F87J11
	#elif defined(__18F87J11)
        #pragma config DEBUG = OFF
        #pragma config XINST = OFF
		#pragma config FCMEN = OFF
        #pragma config FOSC = HS
        #pragma config STVREN = ON
		#pragma config WDTEN = OFF
		#pragma config CP0 = OFF              

	#elif defined(__PIC24F__)
	// Explorer 16 board
	_CONFIG2(FNOSC_PRI & POSCMOD_XT)					// Primary XT OSC
	_CONFIG1(JTAGEN_OFF & FWDTEN_OFF & WDTPS_PS2)		// JTAG off, watchdog timer off
	
	#endif

#if defined(__18F4620) || defined(__18F87J11)
    #pragma romdata longAddressLocation = 0x0E
#endif
ROM unsigned char myLongAddress[8] = {EUI_0,EUI_1,EUI_2,EUI_3,EUI_4,EUI_5,EUI_6,EUI_7};
														//stores mylongaddress at address 0x0E of ROMDATA of the program memory
#if defined(__18F4620) || defined(__18F87J11)
    #pragma romdata
														//defaults to romdata
#endif

#define VERSION "v3.0 "	
#define END_PACKET 0xFD
#define END_RESPONSE 0xFE
														//Version of the Radio Utility Driver....	
/********************** FUNCTION PROTOTYPES *************************/
void BoardInit(void);
void MRF24J40Init(void);
void SetChannel(BYTE channel);
void PHYSetLongRAMAddr(WORD address, BYTE value);
void PHYSetShortRAMAddr(BYTE address, BYTE value);
BYTE PHYGetShortRAMAddr(BYTE address);
BYTE PHYGetLongRAMAddr(WORD address);
void PHYSetDeviceAddress(WORD PANID, WORD shortAddress);
BYTE RSSIlookup(BYTE RSSIval);
void PrintStatus(void);
void ResetMRF24J40(void);
void ControlFunctions(BYTE data);
void Send_Packet(BYTE Packet_Type);
														//Define the prototypes for the functions defined in this file

/********************** CONSTANTS ***********************************/
ROM char PredefinedPacket[] = {0x01,0x08,0xC4,0xFF,0xFF,0xFF,0xFF,0x07,0x00,0x00,0x00,0x00,0x00};
														//Contents of the Predefined packet (IEEE 802.15.4 format)
ROM char PERCommand[] = {0x03,0x08,0xC4,0xFF,0xFF,0xFF,0xFF,0xAA};
ROM unsigned char testShortRegisters [] = {0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,0x0C,0x10,0x11,0x12,0x13,0x14,0x15,0x16,0x1C,0x1D,0x1E,0x1F,0x21,0x22,0x23,0x26,0x27,0x28,0x29,0x2E,0x34,0x3D,0x3F};
ROM unsigned short int testLongRegisters [] = {0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x022A,0x022B,0x022C,0x0230,0x0231,0x0232,0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x0240,0x0241,0x0242,0x0243,0x0244,0x0245,0x0246,0x0247,0x0248,0x0249,0x024A,0x024B,0x024C};
ROM BYTE RSSIlookupTable [] ={90,89,88,88,88,87,87,87,87,86,86,86,86,85,85,85,85,85,84,84,84,84,84,83,83,83,83,82,82,82,82,82,81,81,81,81,81,80,80,80,80,80,80,79,79,79,79,79,78,78,78,78,78,77,77,77,77,77,76,76,76,76,76,75,75,75,75,75,74,74,74,74,74,73,73,73,73,73,72,72,72,72,72,71,71,71,71,71,71,70,70,70,70,70,70,69,69,69,69,69,68,68,68,68,68,68,68,67,67,67,67,66,66,66,66,66,66,65,65,65,65,64,64,64,64,63,63,63,63,62,62,62,62,61,61,61,61,61,60,60,60,60,60,59,59,59,59,59,58,58,58,58,58,57,57,57,57,57,57,56,56,56,56,56,56,55,55,55,55,55,54,54,54,54,54,54,53,53,53,53,53,53,53,52,52,52,52,52,51,51,51,51,51,50,50,50,50,50,49,49,49,49,49,48,48,48,48,47,47,47,47,47,46,46,46,46,45,45,45,45,45,44,44,44,44,43,43,43,42,42,42,42,42,41,41,41,41,41,41,40,40,40,40,40,40,39,39,39,39,39,38,38,38,37,36,35};
//RSSI lookup table from MRF24J40 datasheet
/*ROM BYTE RSSIlookupTable [] ={98,97,96,96,94,94,93,93,92,92,92,92,92,91,91,91,91,90,90,90,89,89,89,89,88,88,88,88,87,87,87,87,86,86,86,85,85,85,85,85,85,85,84,84,84,84,84,84,83,83,83,83,82,82,82,82,82,82,81,81,81,81,80,80,80,80,79,79,79,79,79,78,78,78,78,78,78,77,77,77,77,76,76,76,76,76,76,75,75,75,75,75,75,74,74,74,74,73,73,73,73,73,73,73,72,72,72,72,72,72,71,71,71,71,70,70,70,70,70,69,68,68,68,68,68,68,68,67,67,67,67,66,66,66,66,65,65,65,65,65,64,64,64,64,63,63,63,63,63,63,62,62,62,62,62,61,61,61,61,61,60,60,60,60,60,60,59,59,59,59,59,59,59,58,58,58,58,58,57,57,57,57,57,57,57,56,56,56,56,56,55,55,55,55,55,55,54,54,54,54,53,53,53,53,53,53,52,52,52,52,52,51,51,51,51,50,50,50,49,49,49,49,49,49,49,48,48,48,47,47,47,47,46,46,46,46,46,45,45,45,45,45,44,44,44,44,44,43,43,43,43,43,42,42,42,42,41};
RSSI lookup table based on characterization reports */



/********************** Variables ***********************************/
//Variables accessed globally and in ISR
volatile DWORD_VAL numPacketsRx;
volatile DWORD_VAL PERnum;
volatile DWORD_VAL PERnumFailed;
volatile BOOL foundPERPacket;
volatile BYTE StartSendingPER = 0;
volatile BOOL ReceivingMode;
volatile BOOL RxPacketCountMode;
volatile DWORD PingPongTimeout;
volatile BOOL PingPongBack = FALSE;

//Initialized variables (idata)
BYTE packageGet = 0;
BYTE RxPacket_Count = 0;
BYTE RxPacket_RSSI = 0;
BYTE Print_RxPacket_Count = 0;
BYTE Print_RxPacket_RSSI = 0;
BYTE Header_count = 0;
BYTE PingPongPackage = 100;
BOOL PingPongStart = FALSE;
BOOL PingPongTest = FALSE;
BYTE RxPrintMenu = 20;
BOOL Skip_submenu = FALSE;
BOOL RxMode_verbose = FALSE;
BOOL PacketReceived = FALSE;
BOOL Received_EndPacket = FALSE;
BOOL Received_EndRsp = FALSE;
BOOL Print_RXStats = FALSE;
BOOL MRF24J40_Submenu = FALSE;
BOOL RX_LED_ON =FALSE;
BOOL TX_LED_ON = FALSE;
BYTE PER_TxPackageid = 0;
BYTE PER_RxPackageid = 0xFF;
//Uninitialized varibales (udata)
BYTE numRSSISamples;
BYTE input;
BOOL UseExternal;
BOOL Sync;
BOOL SendSyncRsp;
BYTE packetDelay;
DWORD PingPongSend;
DWORD PingPongGet;
BYTE Packageid;
volatile DWORD RX_RSSI_VAL;
volatile DWORD RX_LQI_VAL;
BYTE AvgRSSI;
BYTE AvgLQI;
BYTE RSSI_dB;
BYTE Testid;

/*********************************************************************
 * Function:        void Delay10KTCYx(BYTE)
 *
 * PreCondition:    None
 *
 * Input:           Required Delay
 *
 * Output:          None
 *
 * Side Effects:   	Delays the next step until the timer expires
 *
 * Overview:        This function introduces necessary delay 
 *
 * Note:            None
 ********************************************************************/
#if defined(__PIC24F__)

void Delay10KTCYx(BYTE d)
{
    DWORD _dcnt;
    _dcnt = ((DWORD)d * (0.01 / (2.0/CLOCK_FREQ)/6));
    while(_dcnt--);    
}

#endif

    
typedef union
{
    BYTE txData[128];
    struct 
    {
        BYTE ShortReg[sizeof(testShortRegisters)];
        BYTE LongReg[sizeof(testLongRegisters)];
    }RegTest;
} TEST_DATA;
    
TEST_DATA testData;

/*********************************************************************
 * Function:        void PrintHeader(ROM char *)
 *
 * PreCondition:    Console needs to be initialized (ConsoleInit())
 *
 * Input:           String that needs to be printer as Header
 *
 * Output:          None
 *
 * Side Effects:    Prints the String on the Console along with the header notation
 *
 * Overview:        This function prints the header
 *
 * Note:            None
 ********************************************************************/

void PrintHeader(ROM char *app)
{
    BYTE i;
    BYTE stringlen = 0;
    
    while( (i = app[stringlen++]) );
    stringlen--;
	//calculate the string len for centering the string *app

    ConsolePutROMString((ROM char*)"\r\n---------------------------------------------------------------\r\n");    
    ConsolePutROMString((ROM char*)"           Microchip MRF24J40 Radio Utility Driver Program         \r\n");
    ConsolePutROMString((ROM char*)"                             Version: ");
    ConsolePutROMString((ROM char*)VERSION);
    ConsolePutROMString((ROM char*)"                         \r\n");   
	ConsolePutROMString((ROM char*)"---------------------------------------------------------------\r\n");    
	//print the header

    ConsolePutROMString(app);
	
    ConsolePutROMString((ROM char *)":\r\n");
	for (i=0; i<stringlen; i++)
	{
	ConsolePutROMString((ROM char *)"-");
	}
	nl();
	//print the end line and move cursor over to new line; can even use nl() function   
}

/*********************************************************************
 * Function:        void PrintSubHeader(ROM char *)
 *
 * PreCondition:    Console needs to be initialized (ConsoleInit())
 *
 * Input:           String (subheader)
 *
 * Output:          Prints the subheader on console
 *
 * Side Effects:    None
 *
 * Overview:        This function prints the char input in subheader format
 *
 * Note:            None
 ********************************************************************/
 
void PrintSubHeader(ROM char *app)
{

BYTE i;
BYTE stringlen = 0;
	while( (i = app[stringlen++]) );
	stringlen--;
	ConsolePutROMString(app);
	
    ConsolePutROMString((ROM char *)":\r\n");
	for (i=0; i<stringlen; i++)
	{
	ConsolePutROMString((ROM char *)"-");
	}
	nl();
	//print the end line and move cursor over to new line   
}

/*********************************************************************
 * Function:        void PrintStatus(void)
 *
 * PreCondition:    Console and MRF24J40 device needs to be initialized (ConsoleInit(), MRF24J40Init())
 *
 * Input:           None
 *
 * Output:          Prints MRF24J40 status on console
 *
 * Side Effects:    None
 *
 * Overview:        This function prints MRF24J40 Status on the console(hyperterminal)
 *
 * Note:            None
 ********************************************************************/


void PrintStatus()
{
BYTE i,j,value,remainder;
DWORD Delay;
PrintSubHeader((ROM char *)"System Status");
												//Channel details
ConsolePutROMString((ROM char *)"Channel = ");
i = PHYGetLongRAMAddr(RFCON0);
i = ((i>>4) + 11);
PrintDigit(i);
												//Transmit Power details
ConsolePutROMString((ROM char *)"\tTX Power = ");
i = PHYGetLongRAMAddr(RFCON3);
j = i;
i = ((i>>3) & 0x07);
j = (j>>6);
if(j == 1)
	value = 10;
else if(j == 2)
	value = 20;
else if(j == 3)
	value = 30;
else value = 0;
if(i == 1)
 	remainder = 5;
else if(i == 2)
	{
	value = value + 1;
	remainder = 2;
	}
else if(i == 3)
	{
	value = value + 1;
	remainder = 9;
	}
else if(i == 4)
	{
	value = value + 2;
	remainder = 8;
	}
else if(i == 5)
	{
	value = value + 3;
	remainder = 7;
	}
else if(i == 6)
	{
	value = value + 4;
	remainder = 9;
	}
else if(i == 7)
	{
	value = value + 6;
	remainder = 3;
	}
else remainder = 0;
if(!(value == 0 && remainder == 0))
ConsolePutROMString((ROM char *)"-");
PrintDigit(value);
ConsolePutROMString((ROM char *)".");
PrintDigit(remainder);
ConsolePutROMString((ROM char *)"dBm + Gain of External PA");
														//Turbo mode details
ConsolePutROMString((ROM char *)"\tTurbo Mode = ");
i = PHYGetShortRAMAddr(READ_BBREG0);
if (i==0)
	ConsolePutROMString((ROM char *)"Off");
else 
	ConsolePutROMString((ROM char *)"On");
nl();
														//CRC checking details
ConsolePutROMString((ROM char *)"H/W CRC Checking = ");
i = PHYGetShortRAMAddr(READ_RXMCR);
if((i & 0x02) == 0)
	ConsolePutROMString((ROM char *)"On");
else
	ConsolePutROMString((ROM char *)"Off");
														//External PA/LNA details
ConsolePutROMString((ROM char*)"\tExternal PA/LNA = ");
i = PHYGetLongRAMAddr(TESTMODE);
if(i == 0x0F)
	ConsolePutROMString((ROM char *)"On");
else
	ConsolePutROMString((ROM char *)"Off");
														//Packet Delay in terms of units
ConsolePutROMString((ROM char*)"\tPacket Delay = ");
Delay = (packetDelay * packetDelay);
PrintDigit(Delay);
ConsolePutROMString((ROM char*)" units");
														
nl();
c(">>");
}

/*********************************************************************
 * Function:        void ResetMRF24J40(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    MRF24J40 device is reset
 *
 * Overview:        Resets all the variables and initializes MRF24J40
 *
 * Note:            None
 ********************************************************************/

void ResetMRF24J40()
{  
	UseExternal = FALSE;
	ReceivingMode = FALSE;
	numPacketsRx.Val = 0;
	PERnumFailed.Val = 0;
	PERnum.Val = 0;
	foundPERPacket = FALSE;
	numRSSISamples = 1;
	packetDelay = 1;				//resets all the variables to reset values
	MRF24J40Init();
	ConsolePutROMString ((ROM char *)"MRF24J40 Device has been reset \r\n");

}

/*********************************************************************
 * Function:        void ControlFuntions(BYTE)
 *
 * PreCondition:    Console needs to be initialized - ConsoleInit()
 *
 * Input:           Ascii key input
 *
 * Output:          Checks whether the key pressed is (Ctrl+s)/(Ctrl+z)/(Ctrl+x)
 *
 * Side Effects:    Ctrl+s - Prints status, Ctrl+z - returns to main menu, Ctrl+x - resets MRF24J40
 *
 * Overview:        Checks whether the key pressed is (Ctrl+s)/(Ctrl+z)/(Ctrl+x), Ctrl+s - Prints status, Ctrl+z - returns to main menu, Ctrl+x - resets MRF24J40
 *
 * Note:            None
 ********************************************************************/

void ControlFunctions(BYTE data)
{
		if(data == 19)						//Ctrl+s - Print Status
					{
					ConsolePut(data);		//Prints Ctrl+s on the screen
					nl();
					PrintStatus();			//Prints Status
					Skip_submenu = TRUE;	//Wait for input at the same level
					return;
					}					
		else if(data == 24)					//Ctrl+x - Resets MRF24J40
					{
					ConsolePut(data);		//Prints Ctrl+x on the screen
					nl();
					ResetMRF24J40();		//Resets MRF24J40
					MRF24J40_Submenu = TRUE;//Go to the main menu
					}
		else if (data == 26)
					{
					ConsolePut(data);		//Ctrl+z - Exit and Return to main menu
					nl();
					MRF24J40_Submenu = TRUE;//Go to main menu
					}
		else
					return;
}
/********************** FUNCTIONS ***********************************/
#if defined(__18F4620) || defined(__18F87J11)
    void main(void)
#elif defined(__PIC24F__)
    int main(void)
#endif
{   
    BYTE power;
    BYTE i;
    #if !defined(__18F4620)
        char buf[32];
    #endif
    
    // decide if it is a hard reset or soft reset
    #if defined(__18F4620) || defined(__18F87J11)
        if((RCON & 0b00000011) != 0b00000011)
    #endif
    {   
        // hard reset, intialize the variables           
        UseExternal = FALSE;
        ReceivingMode = FALSE;
        RCON |= 0b00000011;
    }
    #if defined(__18F4620) || defined(__18F87J11)
        else
        {
            // soft reset, keep them the way they were
            RCON |= 0b00000011;
        }
    #endif
    
    numPacketsRx.Val = 0;
    PERnumFailed.Val = 0;
    PERnum.Val = 0;
    foundPERPacket = FALSE;
    numRSSISamples = 1; 
    packetDelay = 1;
    BoardInit();
								//Complete the board initilization routine 
    ConsoleInit();	
								//Complete the Console Initialization routing
    
								//MainMenu:
    
        nl();
								//prints new line
    
//MRF24J40_Start: 
    MRF24J40Init();
      							//Initialize the MRF                  
MRF24J40_SubMenu:
	MRF24J40_Submenu = FALSE;
   	PrintHeader((ROM char *)"Main Menu");
        c("    (a) Configure MRF24J40\r\n");
        c("    (b) Set the Radio in Receiving Mode\r\n");
        c("    (c) Transmit Predefined Packet Continuously\r\n");
        c("    (d) Transmit Packet Defined by User\r\n");
        c("    (e) Test Low Power Mode\r\n");
        c("    (f) Energy Detection on All Channels\r\n");
        c("    (g) Test Simple Local Oscillator\r\n");
        c("    (h) Test Single Tone Modulation\r\n");
        c("    (i) Test Sequential Transmit Mode for All Channels\r\n");
        c("    (j) PER Test between Two Devices\r\n");
        c("    (k) Ping Pong Test\r\n");
        c("    (l) Dump Values of Transceiver's Registers\r\n");
        c(">>");				//Print Main menu on the screen
MRF24J40_Wait:    
    							//go to this loop if receive UART interrupt bit is not set
    while(!ConsoleIsGetReady())
    {  
		BYTE j;
        RFIE = 1;				//Enable interrupts
        
        if( RF_INT_PIN == 0 )
        {
            RFIF = 1;		
        } 

								//Checking if MRF24J40 card is inserted in to the board
		j = PHYGetShortRAMAddr(READ_ORDER);
		if(j!= 0xFF)
			{
			ConsolePutROMString((ROM char *)"\r\nMRF24J40 Daughter card not inserted");
			ConsolePutROMString((ROM char*)"\r\nInsert Daughter card and Press (Ctrl+X) to Initialize MRF24J40");
			c("\r\n>>");
			while(1)
				{
				if(ConsoleIsGetReady())			//Wait until key is pressed on the console
					{
						input = ConsoleGet();
						if(input == 24)			//Check if key pressed is Ctrl+x
							{
							ConsolePut(input);	//print the Ctrl+x on console
							nl();
							ResetMRF24J40();	//Reset MRF24J40
							goto MRF24J40_SubMenu;	//go to main menu
							}
					}
				}	 
			}
        if( StartSendingPER )
        {
            //goto SendingPER;     
            while(1)
            {
                BYTE i;
                
                LED_2 = 1;

                //tx packets fast
                PHYSetLongRAMAddr(0x000,sizeof(PredefinedPacket));
												//header length
                PHYSetLongRAMAddr(0x001,sizeof(PredefinedPacket));
                								//length of the packet - packet length
                for(i=2;i<(sizeof(PredefinedPacket)+2);i++)
                {
                    PHYSetLongRAMAddr(i, PredefinedPacket[i-2]); /* Dest Address  */
												//load the predefined packet in to TX FIFO
                }
             	PHYSetLongRAMAddr(0x00A, PER_TxPackageid);   
                {
                    unsigned int blah;
                    
                    blah = 0;
                    
                    while(1)
                    {
                        DWORD j;
                        DWORD k;
                        
												//If receive interrupt is received from UART go to main menu
                        if(ConsoleIsGetReady())                   
                           goto MRF24J40_SubMenu;
                        k = ((DWORD)packetDelay)*((DWORD)packetDelay);
                        
                        do
                        {
                            #if defined(__18F4620) || defined(__18F87J11)
                                for(j=0;j<2000;j++)
                            #else
                                for(j=0;j<4000;j++)
                            #endif
                            {
                            }
                        }
                        while(k--);									  //delay between transmission of packets

                        PHYSetShortRAMAddr(WRITE_TXNCON,0b00000001);  //transmit packet without ACK requested 

                        blah++;
                        if((blah % 100) == 0 )
                        {
                            for(j = 0; j < 0xFFFF; j++) {}
                        }
                        if(blah == 1000)
                        {
                            break;									//transmit packtets until 1000 packets are transmitted
                        }
                    }
                }
                
                LED_2 = 0;
                
                StartSendingPER = 0;
                c("\r\nSent 1000 Packets");
				PER_TxPackageid++;
                //goto MRF24J40_SubMenu;                       		//stop transmitting packets
            }  
        }
    }
    
    PingPongTest = FALSE;       
		//Set the PingPongTest bool value to FALSE					//stop ping pong test
	while(!ConsoleIsGetReady());
    input = ConsoleGet();											//wait until key is pressed on the console
    switch(input)
    {

		case 19:	ConsolePut(input);
					nl();
					PrintStatus();									//print status if input is Ctrl+s key
					Skip_submenu = TRUE;
					break;
		case 24:	ConsolePut(input);
					nl();
					ResetMRF24J40();								//reset MRF24J40 if input is Ctrl+x
					break;
		case 26: 	ConsolePut(input);
					nl();											//returns to main menu if input is Ctrl+z
					break;
        case 'a':
        case 'A':
				ConsolePut(input);
          
                PrintSubHeader((ROM char *)"\r\nConfigure MRF24J40");
                c("    (a) Set Operating Channel\r\n");
                c("    (b) Set TX Output Power\r\n");
                c("    (c) Enable/Disable Hardware CRC checking\r\n");
                c("    (d) Configure External PA & LNA\r\n");
                c("    (e) Set TX Delay Between Packets\r\n");
                c("    (f) Set number of Averaged RSSI Samples\r\n");
                c("    (g) Enable/Disable Turbo Mode\r\n");
                c("    (h) Set Ping Pong Test Package Size\r\n");
                c(">>");
																	//prints "Configure MRF24J40" submenu
MRF24J40_ConfigureMenu:
																	
			while(!ConsoleIsGetReady());							//Wait for the input
            input = ConsoleGet();									//read input from the buffer
            
            switch(input)
            {
				case 19:ConsolePut(input);
						nl();
						PrintStatus();								//print status if input is Ctrl+s
						Skip_submenu = TRUE;
						break;
				case 24:ConsolePut(input);
						nl();
						ResetMRF24J40();							//reset MRF24J40 if input is Ctrl+x
						break;
				case 26:ConsolePut(input);
						nl();										
						break;										//exit and return to main menu if input is Ctrl+z
                case 'a':
                case 'A':
						ConsolePut(input);              

						//print the channel sub menu
                        PrintSubHeader((ROM char *)"\r\nSet the Operating Channel");
                        c("    (a) Channel 11\r\n");
                        c("    (b) Channel 12\r\n");
                        c("    (c) Channel 13\r\n");
                        c("    (d) Channel 14\r\n");
                        c("    (e) Channel 15\r\n");
                        c("    (f) Channel 16\r\n");
                        c("    (g) Channel 17\r\n");
                        c("    (h) Channel 18\r\n");
                        c("    (i) Channel 19\r\n");
                        c("    (j) Channel 20\r\n");
                        c("    (k) Channel 21\r\n");
                        c("    (l) Channel 22\r\n");
                        c("    (m) Channel 23\r\n");
                        c("    (n) Channel 24\r\n");
                        c("    (o) Channel 25\r\n");
                        c("    (p) Channel 26\r\n");
                        c(">>");
																	//print the channel menu
MRF24J40_ChannelInput:     
					while(!ConsoleIsGetReady());                   
                    input = ConsoleGet();
                    if(!(((input>=0x41) & (input<=0x50)) | ((input>=0x61) & (input<=0x70))))
                    {
                        ControlFunctions(input);
						if(Skip_submenu)
							{
								Skip_submenu = FALSE;
								goto MRF24J40_ChannelInput;		//if input is not in (a) - (p) wait for the input again
							}
						if(MRF24J40_Submenu)
							goto MRF24J40_SubMenu;
						else
							goto MRF24J40_ChannelInput;
                        
                    }
                    else
                    {
                        BYTE var;
                        ConsolePut(input);						//print input char on the console
						nl();		
						if(input>=0x61)
                        var = ((input-0x61)<<4);
						else
						var = ((input-0x41)<<4);
                        SetChannel(var);						//set the channel using SetChannel() function
                        PrintStatus();							//print status
                        goto MRF24J40_SubMenu;
                    }
                    break;
                    
                case 'b':
                case 'B':ConsolePut(input);
						//Change - These power levels are according to the v0.6 uesr manual. According to Ubec 1.0 these values the fine scale values changed from that of the steps of 1.25

	                    PrintSubHeader((ROM char *)"\r\nSelect the Range of output power");
                        c("    (a) 0 dBm to -10 dBm\r\n");
                        c("    (b) -10 dBm to -20 dBm\r\n");
                        c("    (c) -20 dBm to -30 dBm\r\n");
                        c("    (d) -30 dBm to -40 dBm\r\n");
                        c(">>");
MRF24J40_POWER:          										//print the power values in menu format
					while(!ConsoleIsGetReady());          
                    input = ConsoleGet();						//read input from the console
                    if(((input>=0x61) & (input<=0x64))| ((input>=0x41) & (input<=0x44)))
                    {
						ConsolePut(input);
						if(input>=0x61)
                        power = ((input-0x61)<<6);
						else
						power = ((input-0x41)<<6);
						//Change - These power levels are according to the v0.6 uesr manual. According to Ubec 1.0 these values the fine scale values changed from that of the steps of 1.25

	              	    	PrintSubHeader((ROM char *)"\r\nSelect the fine scale power value");
                        	c("    (a) 0 dBm\r\n");
                        	c("    (b) -0.5 dBm\r\n");
                        	c("    (c) -1.2 dBm\r\n");
                        	c("    (d) -1.9 dBm\r\n");
                        	c("    (e) -2.8 dBm\r\n");
                        	c("    (f) -3.7 dBm\r\n");
                        	c("    (g) -4.9 dBm\r\n");
                        	c("    (h) -6.3 dBm\r\n");
                        	c(">>");
MRF24J40_FSPOWER:											//print the fine scale power values in menu format
						while(!ConsoleIsGetReady());
						input = ConsoleGet();
                    	if(((input>=0x61) & (input<=0x68)) |((input>=0x41) & (input<=0x48)))
						{
							ConsolePut(input);
							nl();
							if(input>=0x61)
                   			input = ((input-0x61)<<3);
							else
							input = ((input-0x41)<<3);
							power = (power | input);
							PHYSetLongRAMAddr(RFCON3,power);                
							PrintStatus();
                        	goto MRF24J40_SubMenu;
						}
						else 
						{
                        	ControlFunctions(input);	//check whether input is Ctrl+s,Ctrl+z or Ctrl+x
    						if(Skip_submenu)
									{						
										Skip_submenu = FALSE;	
										goto MRF24J40_FSPOWER;
									}
							if(MRF24J40_Submenu)
								goto MRF24J40_SubMenu;                   
                        	goto MRF24J40_FSPOWER;	//if not ctrl+z or Ctrl+x wait for fine scale power input
                    	}    
                  	}
                    else
                    {
						ControlFunctions(input);	//check whether input is ctrl+s, Ctrl+z or Ctrl+x
						if(Skip_submenu)
								{
									Skip_submenu = FALSE;
									goto MRF24J40_POWER;
								}
						if(MRF24J40_Submenu)
							goto MRF24J40_SubMenu;

                        goto MRF24J40_POWER;		//if not ctrl+x or ctrl+z wait for power input
                    }    
                    break;
                
                case 'c':
                case 'C':ConsolePut(input);             
                        PrintSubHeader((ROM char *)"\r\nHardware CRC checking");
                        c("    (a) Disable CRC Checking\r\n"); 
                        c("    (b) Enable CRC Checking\r\n");
                        c(">>");					//print hardware crc checking menu
MRF24J40_CRCInput:
					while(!ConsoleIsGetReady());
                    input = ConsoleGet();
                    switch(input)
                    {
						case 19:ConsolePut(input);		//print input on the console
								nl();
								PrintStatus();			//print status as key pressed is ctrl+s
								goto MRF24J40_CRCInput;	//wait for valid input 
								
						case 24:ConsolePut(input);		//print the key pressed on the console
								nl();
								ResetMRF24J40();		//as input is ctrl+x reset MRF24J40
								break;
						case 26:ConsolePut(input);
								nl();
								break;					//exit current menu level and return to main menu
                        case 'a':
						case 'A':
							ConsolePut(input);
							nl();
                            PHYSetShortRAMAddr(WRITE_RXMCR,0x22);
							// Automatic acknowledgement is not transmitted for received packets by TXMAC
							//ERRpkt mode bit is set to 1
							//Promiscuous packet mode bit is set to 1
							PrintStatus();
                            break;
                        case 'b':
						case 'B':
							ConsolePut(input);
							nl();
                            PHYSetShortRAMAddr(WRITE_RXMCR,0x21);
							// Automatic acknowledgement is not transmitted for received packets by TXMAC
							//ERRpkt mode bit is set to 0
							//Promiscuous packet mode bit is set to 1
                            PrintStatus();
                            break;
                        default:
                            goto MRF24J40_CRCInput;        
                    }
                    goto MRF24J40_SubMenu;
                    
                case 'd':
                case 'D':ConsolePut(input);
						nl();
                    UseExternal = TRUE;
                    PHYSetLongRAMAddr(0x22F, 0x0F);		//configures external PA and LNA
                    PrintStatus();						//print the status before returning to main menu
                    goto MRF24J40_SubMenu;

                case 'e':
                case 'E':ConsolePut(input);              
                        PrintSubHeader((ROM char *)"\r\nTransmission Delay Between Packets");
                        c("    (0) no delay\r\n");
                        c("    (1) 1 unit of delay\r\n");
                        c("    (2) 4 units of delay\r\n");
                        c("    ...\r\n");
                        c("    (9) 81 units of delay\r\n");
                        c("    (a) 100 units of delay\r\n");
                        c("    (b) 121 units of delay\r\n");
                        c("    ...\r\n");
                        c("    (z) 1225 units of delay (approx 1 sec)\r\n");
                        c(">>");					//prints transmission delay menu
MRF24J40_DelayInput:
					while(!ConsoleIsGetReady());
                    input = ConsoleGet();
                    if(input<0x30)
                    {
						ControlFunctions(input);	//check if input is ctrl+s,ctrl+x or ctrl+z
						if(Skip_submenu)
							{
								Skip_submenu = FALSE;
								goto MRF24J40_DelayInput;
							}
						if(MRF24J40_Submenu)
							goto MRF24J40_SubMenu;		
                        goto MRF24J40_DelayInput;
                    }
                    ConsolePut(input);
					nl();
                    input -= 0x30;
                    
                    if(input>9)
                    {
						if(input>=31)			//upper case letters
                        input-=0x27;
						else
						input-=0x7;
                    }
                    
                    packetDelay = input;		
                    							//store the input delay in packetDelay variable
					PrintStatus();
                    goto MRF24J40_SubMenu;

                case 'f':
                case 'F':ConsolePut(input);
                        PrintSubHeader((ROM char *)"\r\nRSSI Sample Number");
                        c("    Please enter 2 decimal digits (1-49)\r\n");
                        c("      i.e. - enter 05 for 5 samples\r\n");
                        c(">>");    //print RSSI samples enu
_sampleRSSILoop1:
                    numRSSISamples = 0;                    
                    while(!ConsoleIsGetReady()){}                
                    input = ConsoleGet();
                    switch(input)
                    {
                        case '4':
                            numRSSISamples += 10;
                     								       //fall through
                        case '3':
                            numRSSISamples += 10;
                            								//fall through
                        case '2':
                            numRSSISamples += 10;
                            								//fall through
                        case '1':
                            numRSSISamples += 10;
                            								//fall through
                        case '0':
                            break;
						case 19:ConsolePut(input);
								nl();
								PrintStatus();				//prints status on the screen as key pressed is ctrl+s
								Skip_submenu = TRUE;
								goto _sampleRSSILoop1;
						case 24:ConsolePut(input);
								nl();
								ResetMRF24J40();			//resets MRF24J40 as input key pressed is ctrl+x
								goto MRF24J40_SubMenu;
						case 26:ConsolePut(input);
								nl();					
								goto MRF24J40_SubMenu;		//exits and returns to main menu as input key pressed is ctrl+z
                        default:
                            goto _sampleRSSILoop1;
                    }
                    
                    ConsolePut(input);
                    
_sampleRSSILoop2:                    
                    while(!ConsoleIsGetReady()){}
                
                    input = ConsoleGet();
                    switch(input)
                    {
                        case '9':
                            numRSSISamples += 1;
                            							//fall through
                        case '8':
                            numRSSISamples += 1;
                            							//fall through
                        case '7':
                            numRSSISamples += 1;
                            							//fall through
                        case '6':
                            numRSSISamples += 1;
                            							//fall through                                                                                                              
                        case '5':
                            numRSSISamples += 1;
                            							//fall through
                        case '4':
                            numRSSISamples += 1;
                            							//fall through
                        case '3':
                            numRSSISamples += 1;
                            							//fall through
                        case '2':
                            numRSSISamples += 1;
                            							//fall through
                        case '1':
                            numRSSISamples += 1;
                            							//fall through
                        case '0':
                            break;
						case 19:ConsolePut(input);
								nl();
								PrintStatus();			//print system status as key pressed is ctrl+s
								Skip_submenu = TRUE;
								goto _sampleRSSILoop2;
						case 24:ConsolePut(input);
								nl();
								ResetMRF24J40();		//reset MRF24J40 as key pressed is ctrl+x
								goto MRF24J40_SubMenu;
						case 26:ConsolePut(input);
								nl();
								goto MRF24J40_SubMenu;	//exit and return to maun menu
                        default:
                            goto _sampleRSSILoop2;
                    }
                    
                    ConsolePut(input);
                    
                    if(numRSSISamples == 0)
                    {
      	                c("\r\n\r\ndivide by 0 not allowed.  setting number of samples to 1.\r\n");                
                        numRSSISamples = 1;
                    }
                    
                    c("\r\n\r\n");
                    
                    goto MRF24J40_SubMenu;
                    
                case 'g':
                case 'G':ConsolePut(input);
MRF24J40_Turbo_Mode_submenu:                
                        PrintSubHeader((ROM char *)"\r\nTurbo Mode");
                        c("    (a) Enable Turbo Mode\r\n");
                        c("    (b) Disable Turbo Mode\r\n");
                        c(">>");  				//print turbo mode menu
MRF24J40_TurboInput:
					while(!ConsoleIsGetReady());
                    input = ConsoleGet();
                    switch(input)
                    {
                        case 'a':
                        case 'A':
							ConsolePut(input);
                            PHYSetShortRAMAddr(WRITE_BBREG0, 0x01);
                            PHYSetShortRAMAddr(WRITE_BBREG3, 0x38);
                            PHYSetShortRAMAddr(WRITE_BBREG4, 0x5C);
                            
													//Program the Turbo mode configuration registers

                            PHYSetShortRAMAddr(WRITE_SOFTRST,0x02);
													//Reset the Baseband Circuitry
                            c("\r\nDevice set to turbo mode.\r\n");
                            break;
                           
                       case 'b':
                       case 'B':
							ConsolePut(input);
                            PHYSetShortRAMAddr(WRITE_BBREG0, 0x00);
                            PHYSetShortRAMAddr(WRITE_BBREG3, 0xD8);
                            PHYSetShortRAMAddr(WRITE_BBREG4, 0x9C);
				
													//Program to normal mode 
                            
                            PHYSetShortRAMAddr(WRITE_SOFTRST,0x02);
													//Reset the Baseband Circuitry
                             c("\r\nDevice set to normal mode.\r\n");
                            break;
                          
                        default:
							ControlFunctions(input);	//Check if input is Ctrl+s, ctrl+z or ctrl+x
							if(Skip_submenu)
								goto MRF24J40_TurboInput;
							else if(MRF24J40_Submenu)
								goto MRF24J40_SubMenu;
                            else
	                            goto MRF24J40_Turbo_Mode_submenu;
                    }
                    break;
                    
                    
                case 'h':
                case 'H':ConsolePut(input);
                    {
                        BYTE input0, input1, input2;
PingPongPackageSize_submenu:                        
                            PrintSubHeader((ROM char *)"\r\nSet Ping Pong Package Size");
                            c("Please enter the package size for Ping Pong test in decimal(1-255)\r\n>>");
                        								//prints the ping pong package size mennu details
MRF24J40_PingPong_input0:
                        while(!ConsoleIsGetReady()){}
                        input0 = ConsoleGet();
                        
                        if( input0 < '0' || input0 > '9' )
                        {
                            if( input0 == 0x0D )
                            {
                                c("\r\nPong Pong Package Size not set");
                                break;
                            }
							ControlFunctions(input0);
							if(Skip_submenu)
								{
									Skip_submenu = FALSE;	
									goto MRF24J40_PingPong_input0;
								}
							else if(MRF24J40_Submenu)
									goto MRF24J40_SubMenu;
							else
							{
                            c("\r\nInvalid input: ");
                            PrintChar(input0);
                            goto PingPongPackageSize_submenu;
							}
                        }
						else
							ConsolePut(input0);
MRF24J40_PingPong_input1:
                        while(!ConsoleIsGetReady()){}
                        input1 = ConsoleGet();
                        
                        if( input1 < '0' || input1 > '9' )
                        {
                            if( input1 == 0x0D )
                            {
                                PingPongPackage = input0 - '0';
                                c("\r\nPing Pong Package Size is set to 0x");
                                PrintChar(PingPongPackage);
                                c("\r\n");
                                break;
                            }
                            else
                            {
								ControlFunctions(input1);	//check if input is ctrl+s or ctrl+z or ctrl+x
								if(Skip_submenu)
									{
										Skip_submenu = FALSE;	
										goto MRF24J40_PingPong_input1;
									}
								else if(MRF24J40_Submenu)
									goto MRF24J40_SubMenu;
								else
									{
                            			c("\r\nInvalid input: ");
                            			PrintChar(input1);
                            			goto PingPongPackageSize_submenu;
							
									}
                            }
                        }
						else
							ConsolePut(input1);
MRF24J40_PingPong_input2:
                        while(!ConsoleIsGetReady()){}
                        input2 = ConsoleGet();
                        
                        if( input2 >= '0' && input2 <= '9' )
                        {
							ConsolePut(input2);
                            PingPongPackage = (input0-'0') * 100 + (input1 - '0')* 10 + (input2 - '0');
                        }
                        else
                        {
                            if( input2 == 0x0D )
                            {
                                PingPongPackage = (input0-'0') * 10 + (input1-'0');
                            }
                            else
                            {
                                ControlFunctions(input2);
								if(Skip_submenu)
									{
										Skip_submenu = FALSE;	
										goto MRF24J40_PingPong_input2;
									}
								else if(MRF24J40_Submenu)
									goto MRF24J40_SubMenu;
								else
									{
                            			c("\r\nInvalid input: ");
                            			PrintChar(input2);
                            			goto PingPongPackageSize_submenu;
							
									}
                            }
                        }
                        c("\r\nPing Pong Package Size is set to 0x");
                        PrintChar(PingPongPackage);
                        c("\r\n");
                    }
                    break;
                    
                default:Skip_submenu = TRUE;
                    	break;
            }
			if(!Skip_submenu)
            	break;
			else
				{
				Skip_submenu = FALSE;
				goto MRF24J40_ConfigureMenu;
				}
            
            	case 18:	RxMode_verbose = TRUE;
                case 'b':
                case 'B':	
					{
					BYTE channel_val;
					ConsolePut(input);
                    PHYSetShortRAMAddr(WRITE_RXFLUSH,0x01);
						nl();
                        c("Placing transceiver in receive mode\r\n");
						if(!RxMode_verbose)
						{

							c("Choose one of the following modes to print RX packet statistics\r\n");
							c("    (a) Verbose Mode\r\n");
							c("    (b) Summary Mode\r\n");
MRF24J40_RXMODE:										//prints receive mode menu
							while(!ConsoleIsGetReady());
							input = ConsoleGet();
							ControlFunctions(input);	//check if input is ctrl+s or ctrl+x or ctrl+z
							if(Skip_submenu)
								{
									Skip_submenu = FALSE;
									goto MRF24J40_RXMODE;
								}
							else if(MRF24J40_Submenu)
									goto MRF24J40_SubMenu;
						}
						else
						{
							RxMode_verbose = FALSE;		//default mode is verbose
							input = 'a';
						}
						
						switch(input)
							{
								case 'A':
								case 'a': ReceivingMode = TRUE;
										RxPacketCountMode = FALSE;
										break;			//verbose mode
								case 'B':
								case 'b': RxPacketCountMode = TRUE;
										ReceivingMode = FALSE;
										Header_count = 0;
										channel_val = PHYGetLongRAMAddr(RFCON0);
										channel_val = ((channel_val>>4) + 11);
										break;
								default: goto MRF24J40_RXMODE;	
										break;			//summary mode
							}
						c("RSSI values were calculated from the assumption that LNA is not implemented and antenna gain is 0dB\r\n");				
                        c("Press (Ctrl + Z) to Exit and return to Main menu\r\n");
                        c(">>");

                    #if defined(__18F4620) || defined(__18F87J11)
                        //GIEH = 1;       				//enable interrupts
                    #endif
                    RFIE = 1;
			if(RxPacketCountMode)
				{
					//Initialize the timers
					#if defined(__18F87J11)
					T0CON = 0x87;
														//program the mode and prescaler values
					TMR0H = 0xD9;
					TMR0L = 0xDA;
														//for 1 sec
					INTCONbits.TMR0IE = 0;
														//disable interrupt from the timer
					#elif defined(__PIC24F__)
					T1CON = 0x8030;
														//program the mode and prescaler values
					PR1 = 0x3D09;
														//load the period register with value equivalent to 1 sec
					IEC0bits.T1IE = 0;
														//disable interrupt from the timer
             		#else
					T0CON = 0x87;
														//program the mode and prescaler values
					TMR0H = 0xC2;
					TMR0L = 0xF6;
														//for 1 sec
					INTCONbits.TMR0IE = 0;
														//disable interrupt from the timer
					#endif
				}
                while(1)
                  {
					if( ReceivingMode & PacketReceived )
                    {	
						BYTE i,j, RSSI_VAL;
						i = PHYGetLongRAMAddr(0x300);
						RSSI_VAL = PHYGetLongRAMAddr(0x300+i+2);
						PacketReceived = FALSE;
						if(RxPrintMenu++ == 20)
						{
							c("\r\nRSSI | Packet Length | Packet Data\r\n\r\n");
							RxPrintMenu = 0;
						}
                        PrintChar(RSSI_VAL);
						c(" | ");
                        PrintChar(i);   								//print the packet length
						c(" | ");
                        for(j=1;j<=i;j++)
                        {
                            PrintChar(PHYGetLongRAMAddr(0x300 + j));    //read out the rest of the buffer
                        }
                        c("\r\n");
						
                    }
					if(RxPacketCountMode)
						{
						#if defined(__18F87J11)
							while(!(ConsoleIsGetReady() || INTCONbits.TMR0IF));
							INTCONbits.TMR0IF = 0;
							TMR0H = 0xD9;
							TMR0L = 0xDA;				//timer values
							T0CON = 0x87;
							
						#elif defined(__PIC24F__)
							while(!(ConsoleIsGetReady() || IFS0bits.T1IF));
							IFS0bits.T1IF = 0;
						#else
							while(!(ConsoleIsGetReady() || INTCONbits.TMR0IF));
							INTCONbits.TMR0IF = 0;
							TMR0H = 0xC2;
							TMR0L = 0xF6;				//timer values
							T0CON = 0x87;
						#endif

						if(Header_count == 0)
							{
								ConsolePutROMString((ROM char*)"Channel ");
								PrintDigit(channel_val);
								ConsolePutROMString((ROM char*)"RxPacket Count       RSSI(in Decimal & dBm)   \r\n");
							}
						Header_count++;
						if(Header_count == 12) Header_count = 0;
						c("               ");
						RFIE = 0;
														//disable interrupts
						Print_RxPacket_Count = RxPacket_Count;
						Print_RxPacket_RSSI = RxPacket_RSSI;
						RxPacket_Count = 0;
						RxPacket_RSSI = 0;
						RFIE = 1;
														//Enable interrupts
						PrintDigit(Print_RxPacket_Count);
						c("            ");
						PrintDigit(Print_RxPacket_RSSI);
						Print_RxPacket_RSSI = RSSIlookup(Print_RxPacket_RSSI);
						ConsolePutROMString((ROM char*) "(-");
						PrintDigit(Print_RxPacket_RSSI);
						ConsolePutROMString((ROM char*) "dBm)\r\n");
						
						}
                        if(ConsoleIsGetReady())
                        {
							RFIE = 0;
							input = ConsoleGet();
							
							if((input == 24) | (input == 26))
							{
								ConsolePut(input);
								nl();
								RxPacketCountMode = FALSE;
                            	ReceivingMode = FALSE;
								if(input == 24)
									ResetMRF24J40();			//if input is ctrl+x reset MRF24J40
                            		goto MRF24J40_SubMenu;		//if input is either ctrl+x or ctrl+z exit and return to main menu
							}
							else
							{
								if(input == 19)
									{
									ConsolePut(input);
									PrintStatus();
									}
								RFIE = 1;
                        	}
                    	}
					 }       
                    break;
					}
                case 20:
                case 'c':
                case 'C':ConsolePut(input);
						nl();
                        ConsolePutROMString((ROM char*)"Entering transmit mode.\r\n");
                        c("Press (Ctrl + Z) to Exit and return to Main menu\r\n");
                        c(">>");
                    {
                        BYTE i;
                        
                        PHYSetLongRAMAddr(0x000,sizeof(PredefinedPacket));
                        PHYSetLongRAMAddr(0x001,sizeof(PredefinedPacket));
                        
                        for(i=2;i<(sizeof(PredefinedPacket)+2);i++)
                        {
                            PHYSetLongRAMAddr(i, PredefinedPacket[i-2]); 
                        }
                    }

                    PHYSetShortRAMAddr(WRITE_TXNCON,0b00000001);  //transmit packet without ACK requested 
					packetDelay = 2;	
        
                    while(1)
                    {
                        unsigned int j;
                        DWORD k;
                        
                        if(ConsoleIsGetReady())
                        {
							input = ConsoleGet();
							if((input == 24) | (input == 26))
							{
								ConsolePut(input);
								nl();
								if(input == 24)
									ResetMRF24J40();
									packetDelay = 1;
                                	goto MRF24J40_SubMenu;
                            }
							else
							{
								if(input == 19)
									PrintStatus();
                        	}
						}

                        k = ((DWORD)packetDelay)*((DWORD)packetDelay);
																//insert delay between consecutive packets that are being transmitted in continuous mode
                        
                        do
                        {
                            for(j=0;j<400;j++)
                            {
                            }
                        }
                        while(k--);

                        PHYSetShortRAMAddr(WRITE_TXNCON,0b00000001);  //transmit packet without ACK requested 
                    }

                    break;
                
                
                case 'd':
                case 'D':ConsolePut(input);
						nl();
                        c("Please enter the packet you want to transmit.\r\n");
                        c("Use all uppercase HEX values starting from the frame control (no length)\r\n");
                        c(" Press '=' when you are done.\r\n");
                        c(">>");							//print the instructions to the user 
                    {
                        BYTE pingPong;
                        BYTE txPtr;
                        BYTE A,B;
						BYTE FC_1, FC_2, FrameType, temp, temp2; 
						BOOL Incorrect_format;
MRF24J40_read_packet:                        
                        txPtr = 0;
                        pingPong = 0;
                        
                        while(1)
                        {
MRF24J40_read_tx:
                            while(!ConsoleIsGetReady()){}    
                            input = ConsoleGet();
                            ControlFunctions(input);			//check whether the input key pressed is either Ctrl+s or ctrl+z or ctrl+x
							if(Skip_submenu)
									{
										Skip_submenu = FALSE;
										goto MRF24J40_read_tx;
									}
							if(MRF24J40_Submenu)
										goto MRF24J40_SubMenu;
                            switch(input)
                            {
                                case '0':
                                case '1':
                                case '2':
                                case '3':
                                case '4':
                                case '5':
                                case '6':
                                case '7':
                                case '8':
                                case '9':
                                    ConsolePut(input);
                                    B = (input - '0');
                                    break;
                                case 'A':
                                case 'B':
                                case 'C':
                                case 'D':
                                case 'E':
                                case 'F':
                                    ConsolePut(input);
                                    B = (input - 'A' + 0x0A);
                                    break;
								case 'a':
                                case 'b':
                                case 'c':
                                case 'd':
                                case 'e':
                                case 'f':
                                    ConsolePut(input);
                                    B = (input - 'a' + 0x0A);
									break;
                                case '=':
																//Verifying whether Tx packet is according to IEEE 802.15.4 format
									
									FC_1 = testData.txData[0];
									FC_2 = testData.txData[1];
									FrameType = (FC_1 & 0x07);
									Incorrect_format = TRUE;
							
									switch(FrameType)
										{
										case 0:					//case for beacon frame
												temp = (FC_2 & 0x0C);
												temp2 = (FC_2 & 0xC0);
												if(temp!= 0x00)
														{
														c("\r\nDestination address mode must be 0x00\r\n");
														break;
														}
																//Dst address info present
												else if((temp2 == 0x00) | (temp2 == 0x01))
														{
														c("\r\nSource address mode cannot be 0x00 or 0x01\r\n");
																//Src addressing mode incorrect
														break;
														}
												else if((temp2 == 0x80) & (txPtr<9))
														{
														c("\r\nAddress information doesn't match the addressing modes specified in frame control field\r\n");
																//In sufficient packet length (2+1+4+2)
														break;
														}
												else if((temp2 == 0xC0) & (txPtr<15))
														{
														c("\r\nAddress information doesn't match the addressing modes specified in frame control field\r\n");
																//Insufficient packet length
														break;
														}
												else	Incorrect_format = FALSE;					 
												break; 
										case 1:	temp = ((FC_2 & 0x0C)>>2);	//dst addr mode
												temp2 = ((FC_2 & 0xC0)>>6);	//src addr mode
												if((temp == 0x01) | (temp2 == 0x01))
														{
														c("\r\n Address mode 0x01 cannot be used\r\n");
																	//if src addr mode and dst addr mode = rsvd
													break;
														}
												else if(((FC_1 & 0x40) == 0x00) & (temp2 == 0x00) & (temp == 0x00))
														{
														c("\r\n Used incorrect addressing modes\r\n");
																	//if Intra-PAN = 0 and either src addr mode / dst addr mode do not contain PAN ID
													break;
														}
												else if(((FC_1 & 0x40) == 0x40) & (temp2 == 0x00) & (temp == 0x02) & (txPtr<7))
														{
														c("\r\nAddress information doesn't match the addressing modes specified in frame control field\r\n");
																	//If Intra-PAN = 1 then length check based on src addr mode and dst addr mode
													break;
														}
												else if(((FC_1 & 0x40) == 0x40) & (temp2 == 0x02) & (temp == 0x00) & (txPtr<7))	
														{
														c("\r\nAddress information doesn't match the addressing modes specified in frame control field\r\n");
																	//If Intra-PAN = 1 then length check based on src addr mode and dst addr mode
													break;
														}
												else if(((FC_1 & 0x40) == 0x40) & (temp2 == 0x00) & (temp == 0x03) & (txPtr<13))	
														{
														c("\r\nAddress information doesn't match the addressing modes specified in frame control field\r\n");
																	//If Intra-PAN = 1 then length check based on src addr mode and dst addr mode
													break;
														}
												else if(((FC_1 & 0x40) == 0x40) & (temp2 == 0x03) & (temp == 0x00) & (txPtr<13))	
														{
														c("\r\nAddress information doesn't match the addressing modes specified in frame control field\r\n");
																	//If Intra-PAN = 1 then length check based on src addr mode and dst addr mode
													break;
														}
												else if(((FC_1 & 0x40) == 0x00) & (temp2 == 0x02) & (temp == 0x02) & (txPtr<8))	
														{
														c("\r\nAddress information doesn't match the addressing modes specified in frame control field\r\n");
																	//If Intra-PAN = 0 then length check based on src addr mode and dst addr mode
													break;
														}
												else if(((FC_1 & 0x40) == 0x40) & (temp2 == 0x02) & (temp == 0x02) & (txPtr<6))	
														{
														c("\r\nAddress information doesn't match the addressing modes specified in frame control field\r\n");
																	//If Intra-PAN = 1 then length check based on src addr mode and dst addr mode
													break;
														}
												else if(((FC_1 & 0x40) == 0x00) & (temp2 == 0x02) & (temp == 0x03) & (txPtr<14))	
														{
														c("\r\nAddress information doesn't match the addressing modes specified in frame control field\r\n");
																	//If Intra-PAN = 0 then length check based on src addr mode and dst addr mode
													break;
														}
												else if(((FC_1 & 0x40) == 0x40) & (temp2 == 0x02) & (temp == 0x03) & (txPtr<12))	
														{
														c("\r\nAddress information doesn't match the addressing modes specified in frame control field\r\n");
																	//If Intra-PAN = 1 then length check based on src addr mode and dst addr mode
													break;
														}
												else if(((FC_1 & 0x40) == 0x00) & (temp2 == 0x03) & (temp == 0x02) & (txPtr<14))	
														{
														c("\r\nAddress information doesn't match the addressing modes specified in frame control field\r\n");
																	//If Intra-PAN = 0 then length check based on src addr mode and dst addr mode
													break;
														}
												else if(((FC_1 & 0x40) == 0x40) & (temp2 == 0x03) & (temp == 0x02) & (txPtr<12))	
														{
														c("\r\nAddress information doesn't match the addressing modes specified in frame control field\r\n");
																	//If Intra-PAN = 1 then length check based on src addr mode and dst addr mode
													break;
														}
												else if(((FC_1 & 0x40) == 0x00) & (temp2 == 0x03) & (temp == 0x03) & (txPtr<20))	
														{
														c("\r\nAddress information doesn't match the addressing modes specified in frame control field\r\n");
																	//If Intra-PAN = 0 then length check based on src addr mode and dst addr mode
													break;
														}
												else if(((FC_1 & 0x40) == 0x40) & (temp2 == 0x03) & (temp == 0x03) & (txPtr<18))	
														{
														c("\r\nAddress information doesn't match the addressing modes specified in frame control field\r\n");
																	//If Intra-PAN = 1 then length check based on src addr mode and dst addr mode
													break;
														}
												else Incorrect_format = FALSE;
												break;
										case 2: 					//ack request
												if((txPtr < 3) | (txPtr >3))
												{	
												c("\r\nIncorrect length for an Acknowledgment frame\r\n");
													break;
												}
												else 
													Incorrect_format = FALSE;
												break;
										case 3:						//mac cmd frame
												temp = ((FC_2 & 0x0C)>>2);	//dst addr mode
												temp2 = ((FC_2 & 0xC0)>>6);	//src addr mode
												if((temp == 0x01) | (temp2 == 0x01))
														{
														c("\r\n Address mode 0x01 cannot be used\r\n");
																	//if src addr mode and dst addr mode = rsvd
													break;
														}
												else if(((FC_1 & 0x40) == 0x00) & (temp2 == 0x00) & (temp == 0x00))
														{
														c("\r\nIncorrect addressing modes\r\n");
																	//if Intra-PAN = 0 and either src addr mode / dst addr mode do not contain PAN ID
													break;
														}
												else if(((FC_1 & 0x40) == 0x40) & (temp2 == 0x00) & (temp == 0x02) & (txPtr<8))	
														{
														c("\r\nAddress information doesn't match the addressing modes specified in frame control field\r\n");
																	//If Intra-PAN = 1 then length check based on src addr mode and dst addr mode
													break;
														}
												else if(((FC_1 & 0x40) == 0x40) & (temp2 == 0x02) & (temp == 0x00) & (txPtr<8))	
														{
														c("\r\nAddress information doesn't match the addressing modes specified in frame control field\r\n");
																	//If Intra-PAN = 1 then length check based on src addr mode and dst addr mode
													break;
														}
												else if(((FC_1 & 0x40) == 0x40) & (temp2 == 0x00) & (temp == 0x03) & (txPtr<14))	
														{
														c("\r\nAddress information doesn't match the addressing modes specified in frame control field\r\n");
																	//If Intra-PAN = 1 then length check based on src addr mode and dst addr mode
													break;
														}
												else if(((FC_1 & 0x40) == 0x40) & (temp2 == 0x03) & (temp == 0x00) & (txPtr<14))	
														{
														c("\r\nAddress information doesn't match the addressing modes specified in frame control field\r\n");
																	//If Intra-PAN = 1 then length check based on src addr mode and dst addr mode
													break;
														}
												else if(((FC_1 & 0x40) == 0x00) & (temp2 == 0x02) & (temp == 0x02) & (txPtr<9))	
														{
														c("\r\nAddress information doesn't match the addressing modes specified in frame control field\r\n");
																	//If Intra-PAN = 0 then length check based on src addr mode and dst addr mode
													break;
														}
												else if(((FC_1 & 0x40) == 0x40) & (temp2 == 0x02) & (temp == 0x02) & (txPtr<7))	
														{
														c("\r\nAddress information doesn't match the addressing modes specified in frame control field\r\n");
																	//If Intra-PAN = 1 then length check based on src addr mode and dst addr mode
													break;
														}
												else if(((FC_1 & 0x40) == 0x00) & (temp2 == 0x02) & (temp == 0x03) & (txPtr<15))	
														{
														c("\r\nAddress information doesn't match the addressing modes specified in frame control field\r\n");
																	//If Intra-PAN = 0 then length check based on src addr mode and dst addr mode
													break;
														}
												else if(((FC_1 & 0x40) == 0x40) & (temp2 == 0x02) & (temp == 0x03) & (txPtr<13))	
														{
														c("\r\nAddress information doesn't match the addressing modes specified in frame control field\r\n");
																	//If Intra-PAN = 1 then length check based on src addr mode and dst addr mode
													break;
														}
												else if(((FC_1 & 0x40) == 0x00) & (temp2 == 0x03) & (temp == 0x02) & (txPtr<15))	
														{
														c("\r\nAddress information doesn't match the addressing modes specified in frame control field\r\n");
																	//If Intra-PAN = 0 then length check based on src addr mode and dst addr mode
													break;
														}
												else if(((FC_1 & 0x40) == 0x40) & (temp2 == 0x03) & (temp == 0x02) & (txPtr<13))	
														{
														c("\r\nAddress information doesn't match the addressing modes specified in frame control field\r\n");
																	//If Intra-PAN = 1 then length check based on src addr mode and dst addr mode	
													break;
														}
												else if(((FC_1 & 0x40) == 0x00) & (temp2 == 0x03) & (temp == 0x03) & (txPtr<21))	
														{
														c("\r\nAddress information doesn't match the addressing modes specified in frame control field\r\n");
																	//If Intra-PAN = 0 then length check based on src addr mode and dst addr mode
													break;
														}
												else if(((FC_1 & 0x40) == 0x40) & (temp2 == 0x03) & (temp == 0x03) & (txPtr<19))	
														{
														c("\r\nAddress information doesn't match the addressing modes specified in frame control field\r\n");
																	//If Intra-PAN = 1 then length check based on src addr mode and dst addr mode
													break;
														}
												else Incorrect_format = FALSE;
												break;
										case 4:
										case 5:
										case 6:
										case 7:	c("\r\n Incorrect frame type is being used\r\n");
												break;
										default: break;
										}
									if(Incorrect_format)
										{
										c("Please reenter the frame in correct format\r\n");
										c(">>");
										goto MRF24J40_read_packet;
										}
                                   
                                        c("\r\nPacket Transmission - Success\r\nReturning to the main menu.\r\n\r\n");
                                   {
                                        BYTE i;
                                        
                                        for(i=0;i<txPtr;i++)
                                        {
                                            PHYSetLongRAMAddr(i+2, testData.txData[i]); /* Dest Address  */
                                        }
                                    }
                                    PHYSetLongRAMAddr(0x000,txPtr);
                                    PHYSetLongRAMAddr(0x001,txPtr);
                                    
                                    PHYSetShortRAMAddr(WRITE_TXNCON,0b00000001);  //transmit packet without ACK requested 
                                                                        
                                    goto MRF24J40_SubMenu;
                                    break;
                                default:
                                    goto MRF24J40_read_tx;
                                    break;
                            }
                            if(pingPong == 1)
                            {
                                testData.txData[txPtr++] = A+B;
                                pingPong = 0;
                            }
                            else
                            {
                                A = B<<4;
                                pingPong = 1;
                            }
                        }
                    }
                    goto MRF24J40_SubMenu;
                
                case 'e':
                case 'E':ConsolePut(input);                
					PrintSubHeader((ROM char *)"\r\nTest Low Power Mode");
                        c("    (a) Place MRF in Sleep Mode\r\n");
						c("    (b) Wake up the MRF from Sleep Mode\r\n");
						c("    (c) Place MRF in Reset\r\n");
                        c(">>");					//print low power mode menu
                    
MRF24J40_LPInput:
					while(!ConsoleIsGetReady());
                    input = ConsoleGet();
					ControlFunctions(input);
					if(Skip_submenu)
						{
						Skip_submenu = FALSE;
						goto MRF24J40_LPInput;
						}
					if(MRF24J40_Submenu)
						goto MRF24J40_SubMenu;
                    switch(input)
                    {
                        case 'a':
                        case 'A':
							ConsolePut(input);
                                c("\r\nMRF24J40 device set to sleep mode\r\n");
                            #if defined(__18F4620) || defined(__18F87J11)
                                RC1 = 0;
                                TRISC1 = 0;
                            #else
                                PHY_WAKE = 0;
                                PHY_WAKE_TRIS = 0;
                            #endif
                            
                            PHYSetShortRAMAddr(WRITE_SOFTRST,0x04);
                            
                            PHYSetShortRAMAddr(WRITE_RFCTL,0x00);
                            						//change - Doing RF state machine reset first 
                            PHYSetShortRAMAddr(WRITE_WAKECON,0x80);
													//WAKECON = 0x80 Enable immediate wakeup mode
                            PHYSetShortRAMAddr(WRITE_RXFLUSH,0b01100000);
													// RXFLUSH = 0x60 Enabling WAKE pin and setting polarity high
                            PHYSetShortRAMAddr(WRITE_SLPACK,0x80);
													//SLPACK = 0x35 Put MRF24J40 to sleep mode
                            break;
                        case 'b':
                        case 'B':
							ConsolePut(input);
                                c("\r\nWaking up the MRF24J40\r\n");
                            PHY_WAKE = 1;
                            break;
                        
                        case 'c':
                        case 'C':
							ConsolePut(input);
                            MRF24J40Init();
                            c("\r\nMRF24J40 has been reset \r\n");
                            break;
                        default:
                            goto MRF24J40_LPInput;
                        
                    }
                    break;
                
                case 'f':
                case 'F':ConsolePut(input);
                    {
                        BYTE currentChannel = 0;
                        BYTE i, j;
                        WORD k;
                        BYTE maxRSSI;
						BYTE maxRSSI_dB;
                        BYTE originalChannel = PHYGetLongRAMAddr(RFCON0);
						c("\r\nMax Spectral energy values will be displayed for each channel in RSSI decimal values and dBm units:\r\n");
						c("Values were calculated from the assumption that LNA is not implemented and antenna gain is 0dB\r\n");

                        if(UseExternal)
                        {
                            PHYSetLongRAMAddr(0x22F, 0x08);
                            PHYSetShortRAMAddr(WRITE_TRISGPIO, 0b00001111);
                            PHYSetShortRAMAddr(WRITE_GPIO, 0b00001100); 
								//Setting external PA and LNA mode
                        }

						nl(); nl();
                        for(i = currentChannel; i <= 15; i++)
                        {
                            BYTE RSSIcheck;
                            maxRSSI = 0;
                            // set channel
                            j = i << 4;
                            SetChannel(j);
                            for(k = 0; k < 0xFFF; k++); 
                            
                            for(j = 0; j < 200; j++)
                            {
                                // set RSSI request
                                PHYSetShortRAMAddr(WRITE_BBREG6, 0x80);
                                
                                // check RSSI
                                RSSIcheck = PHYGetShortRAMAddr(READ_BBREG6);
                                while((RSSIcheck & 0x01) != 0x01 || (RSSIcheck & 0x80))
                                {
                                    RSSIcheck = PHYGetShortRAMAddr(READ_BBREG6);
									//Read RSSI value from the RSSI register
                                    
                                }
                                RSSIcheck = PHYGetLongRAMAddr(0x210);
                                
                                if( RSSIcheck > maxRSSI )
                                {
                                    maxRSSI = RSSIcheck;
                                }
                                for(k = 0; k < 0xFFF; k++);
                                
                            }
							maxRSSI_dB = RSSIlookup(maxRSSI);
							//convert the integer values of RSSI in to dB

                            ConsolePutROMString((ROM char *)"Channel ");

							if(i>=9)
							{
								ConsolePut('2');
							}
							else
							{
								ConsolePut('1');
							}

							{
								unsigned a;

								a = 0x30 + ((i+1)%10);
								ConsolePut(a);
							}
                            ConsolePutROMString((ROM char *)": ");
                            j = maxRSSI/5;
                            for(k = 0; k < j; k++)
                            {
                                ConsolePut('-');
                            }
                            c(" ");
							PrintDigit(maxRSSI);
							ConsolePutROMString((ROM char *)" (-");
							PrintDigit(maxRSSI_dB);
							ConsolePutROMString((ROM char *)"dBm)");
                            ConsolePutROMString((ROM char *)"\r\n");
                        }
                        c("Press (Ctrl+z) to return to the main menu\r\n");
Wait_Input:             while(!ConsoleIsGetReady()){}
                        input = ConsoleGet();
						if (input!=26)
							goto Wait_Input;            	
										//check if input is ctrl+z
						else
							ConsolePut(input);
                        if(UseExternal)
                        {
                            PHYSetShortRAMAddr(WRITE_GPIO, 0);
                            PHYSetShortRAMAddr(WRITE_TRISGPIO, 0b00000000);
                            PHYSetLongRAMAddr(0x22F, 0x0F);
                        }
                        
                        PHYSetLongRAMAddr(RFCON0, originalChannel);
                        PHYSetShortRAMAddr(WRITE_RFCTL,0x04);
                        PHYSetShortRAMAddr(WRITE_RFCTL,0x00);
						PHYSetShortRAMAddr(WRITE_BBREG6, 0x40);    //Set to RSSI Mode 2 (default value)
                        goto MRF24J40_SubMenu;
                    
                    }
                    break;                
                    
                case 'g':
                case 'G':ConsolePut(input);
                        c("\r\nRunning in CW mode\r\n");
                        c("Press (Ctrl + Z) to Exit and return to Main menu\r\n");
                        c(">>");					//testing simple local oscillator mode
                           
                    if(UseExternal)
                    {
                        PHYSetLongRAMAddr(0x22F, 0x08);
                        PHYSetShortRAMAddr(WRITE_TRISGPIO, 0b00001111);
                        PHYSetShortRAMAddr(WRITE_GPIO, 0b00000011);
													//if using external PA and LNA
                    } 
                    
                    PHYSetShortRAMAddr(WRITE_RFCTL,0b00000010);  //force transmit
                    		
                    while(1)						//set TX bit in RFCTL
                    {
                        if(ConsoleIsGetReady())
                        {
                            input = ConsoleGet();
							
							if((input == 24) | (input == 26))		//check if input is ctrl+x or ctrl+z
							{
							ConsolePut(input);
							nl();
							if(input == 24)
								ResetMRF24J40();					//if ctrl+x then reset MRF24J40 and return to main menu
                            if(UseExternal)
                            {
                                PHYSetShortRAMAddr(WRITE_RFCTL,0b00000000);
                                PHYSetShortRAMAddr(WRITE_GPIO, 0b00000000);
                                PHYSetShortRAMAddr(WRITE_TRISGPIO, 0b00000000);
                                PHYSetLongRAMAddr(0x22F, 0x0F);
                            }
                            
                                goto MRF24J40_SubMenu;
							}
                        }
                    }
                    break;                
                
                
                case 'h':
                case 'H':	ConsolePut(input);
                    {
                        BYTE currentChannel = CHANNEL_11;
                        BOOL bSweeping;             
						PrintSubHeader((ROM char*)"\r\nTest Single Tone Modulation");		       
                        c("    (a) Single Channel");
                        c("\r\n    (b) Sweeping Channels");
                        c("\r\n>>");				//Single tone modulation menu
MRF24J40_STInput:           
						while(!ConsoleIsGetReady());             
                        input = ConsoleGet();
						ControlFunctions(input);
						if(Skip_submenu)
							{
							Skip_submenu = FALSE;
							goto MRF24J40_STInput;
							}
						if(MRF24J40_Submenu)
							goto MRF24J40_SubMenu;
                        switch(input)
                        {
                            case 'a':
                            case 'A':
								ConsolePut(input);
                                bSweeping = FALSE;
                                break;
                                
                            case 'b':
                            case 'B':
								ConsolePut(input);
                                bSweeping = TRUE;
                                break;
                                
                            default:
                                goto MRF24J40_STInput; 
                        }
                        
                        c("\r\nEntering Single Tone Test Mode");
 						c("\r\n Press (Ctrl + Z) to Exit and return to the Main menu.");
						c("\r\n>>");

                        if(UseExternal)
                        {
                            PHYSetLongRAMAddr(0x22F, 0x08);
                            PHYSetShortRAMAddr(WRITE_TRISGPIO, 0b00001111);
                            PHYSetShortRAMAddr(WRITE_GPIO, 0b00000011);
                        }    

                        do
                        {   
	                      if(ConsoleIsGetReady())
    	                    {
                            	input = ConsoleGet();
							
								if((input == 24) | (input == 26))
									{
										ConsolePut(input);
										nl();
										if(input == 24)
											ResetMRF24J40();
										goto MRF24J40_SubMenu;
									}
							}
                            if( bSweeping )
                            {
                                SetChannel(currentChannel);
                                currentChannel += 0x10;
                            }

                            PHYSetShortRAMAddr(WRITE_RFCTL,0b00000010);
                            PHYSetLongRAMAddr(TESTMODE,0x0d); 
                            
                            Delay10KTCYx(255);
							     if(ConsoleIsGetReady())
    	                    {
                            	input = ConsoleGet();
							
								if((input == 24) | (input == 26))
									{
										ConsolePut(input);
										nl();
										if(input == 24)
											ResetMRF24J40();
										goto MRF24J40_SubMenu;
									}
							}
                            Delay10KTCYx(255);
                            
                        } while(1);
                        
                    }
                    break;
                
                case 'i':
                case 'I':ConsolePut(input);
                    {
                        BYTE currentChannel = CHANNEL_11;
                        c("\r\nEntering Sequential Transmit Mode");
 						c("\r\nPress (Ctrl + Z) to Exit and return to the Main menu\r\n");
						c(">>");
                        
                        while(1)
                        {
                            WORD i;
                            WORD j;
                            
                            SetChannel(currentChannel);
                            currentChannel += 0x10;
                            
//                            PHYSetLongRAMAddr(RFCON3,0x00);
                            #if defined(__18F4620) || defined(__18F87J11)
                                //GIEH = 1;
                            #endif
                            RFIE = 1;
                            
                            PHYSetLongRAMAddr(0x000, 12); /* packet header length */
                            PHYSetLongRAMAddr(0x001, 16); /* total packet length (not including the FCS/CRC or length) */
                            PHYSetLongRAMAddr(0x002, 0x00); /* data byte */
                            PHYSetLongRAMAddr(0x003, 0x01); /* data byte */
                            PHYSetLongRAMAddr(0x004, 0x02); /* data byte */
                            PHYSetLongRAMAddr(0x005, 0x03); /* data byte */
                            PHYSetLongRAMAddr(0x006, 0x04); /* data byte */
                            PHYSetLongRAMAddr(0x007, 0x05); /* data byte */
                            PHYSetLongRAMAddr(0x008, 0x06); /* data byte */
                            PHYSetLongRAMAddr(0x009, 0x07); /* data byte */
                            PHYSetLongRAMAddr(0x00a, 0x08); /* data byte */
                            PHYSetLongRAMAddr(0x00b, 0x09); /* data byte */
                            PHYSetLongRAMAddr(0x00c, 0x0a); /* data byte */
                            PHYSetLongRAMAddr(0x00d, 0x0b); /* data byte */
                            PHYSetLongRAMAddr(0x00e, 0x0c); /* data byte */
                            PHYSetLongRAMAddr(0x00f, 0x0d); /* data byte */
                            PHYSetLongRAMAddr(0x010, 0x0e); /* data byte */
                            PHYSetLongRAMAddr(0x011, 0x0f); /* data byte */
                            
                            for (i=0;i<400;i++)
	                        {
		                        PHYSetShortRAMAddr(WRITE_TXNCON,0b00000001);
		                        for(j = 0; j < 1000; j++) ;
								if(ConsoleIsGetReady())
                        		{
                        		input = ConsoleGet();
								
								if((input == 24) | (input == 26))
									{	ConsolePut(input);
										nl();	
									if(input == 24)
										ResetMRF24J40();
									goto MRF24J40_SubMenu;
									}
								}
		                    }
							
							
                        }
                    }
                    break;

                
                case 'j':
                case 'J':ConsolePut(input);
						nl();
						c("Starting PER test\r\n");
						c("Press (Ctrl+z) to Exit and Return to Main Menu\r\n");
                    {
                        BYTE i;
                        
                        PHYSetLongRAMAddr(0x000,sizeof(PERCommand));
                        PHYSetLongRAMAddr(0x001,sizeof(PERCommand));
                                    
                        for(i=2;i<(sizeof(PERCommand)+2);i++)
                        {
                            PHYSetLongRAMAddr(i, PERCommand[i-2]); 
                        }
                        
                        //GIEH = 1;       //enable interrupts
                        RFIE = 1;
                        PHYSetShortRAMAddr(WRITE_TXNCON,0b00000001);
                        
                        numPacketsRx.Val = 0;
                         
                         numPacketsRx.Val = 0;
                         while(1)
                         {
                             WORD k;
                             if( RF_INT_PIN == 0 )
                             {
                                 RFIF = 1;
                             }
                                                          if(ConsoleIsGetReady())
                             {
								input = ConsoleGet();
								if((input == 24) | (input == 26))
								{
								ConsolePut(input);
								nl();
								if(input == 24)
									ResetMRF24J40();
                                goto MRF24J40_SubMenu;                               
								break;
                             	}
                         	}
							if(PER_RxPackageid != Testid)
							{
							WORD count;
							PER_RxPackageid = Testid;

							
							if(numPacketsRx.Val>2)
							{
							count = numPacketsRx.Val - 1;
							numPacketsRx.Val = 1;
                         //GIEH = 0;       //enable interrupts
                        // RFIE = 0;
                         
                         //print results
                         #if defined(__18F4620) || defined(__18F87J11)
                             printf((far const ROM char*)"\t%d packets received of 1000 ",count);
                         #else
                            sprintf(buf, (ROM char*)"\t%d packets received out of 1000 ",count);
                            c(buf);
                         #endif
						c("\t");
						ConsolePutROMString((ROM char*)"PER = ");
						i = (count/100);
						if(i != 0) 
							{
								PrintDigit(i);
								count = (count%100);
							}
						i = (count/10);
						if(i != 0)
							{
								PrintDigit(i);
								count = (count%10);
							}
						else
							ConsolePut('0');
						ConsolePutROMString((ROM char*)".");
						i = count;
						PrintDigit(i);					//Printing PER value in decimal - float
						ConsolePutROMString((ROM char*)"%");
						nl();
						}//end count>2 
						
						}
						for(k = 0; k < 50; k++) {}
                             #if defined(__18F4620) || defined(__18F87J11)
                                    printf((far const ROM char*)"\t%d packets received\r",numPacketsRx.word.LW);
                             #else
                                sprintf(buf, (ROM char*)"\t%d packets received\r",numPacketsRx.word.LW);
                                c(buf);
                             #endif

                    }
				}	
                    break;
                
                              
                case 'k':
                case 'K':ConsolePut(input);
                    {      
						BYTE display_count = 0, replay_count = 0;
						Sync = FALSE;
						SendSyncRsp = FALSE;
						PrintSubHeader((ROM char*)"\r\n Ping Pong Test");          
                        c("    (a) Receive Ping Pong Test");
                        c("\r\n    (b) Start Ping Pong Test");
                        c("\r\n>>");   
PingPongTest_Input:		while(!ConsoleIsGetReady());
                        input = ConsoleGet();
						ControlFunctions(input);
						if(Skip_submenu)
							{
							Skip_submenu = FALSE;
							goto PingPongTest_Input;
							}
						if(MRF24J40_Submenu)
							goto MRF24J40_SubMenu;
                        switch(input)
                        {
                            case 'a':
                            case 'A':
                                if( packetDelay < 2 )
                                {
                                    packetDelay = 2;
                                }
                                PingPongTest = TRUE;
                                PingPongBack = FALSE;
                                
                                PingPongSend = 0;
                                PingPongGet = 0;
                                packageGet = 0;
								RX_RSSI_VAL = 0;
								RX_LQI_VAL = 0;
                                PingPongTimeout = 0;
                                PingPongStart = FALSE;
                                Packageid = 1;
                                c("\r\nPress Any Key to Return to the Main Menu.");
								
                                while(1)
                                {   
                                    if(!ConsoleIsGetReady())
                                    {
										if(SendSyncRsp)
												{
													PHYSetLongRAMAddr(0x000,sizeof(PredefinedPacket));
                                                    PHYSetLongRAMAddr(0x001,sizeof(PredefinedPacket));
                                                    PHYSetLongRAMAddr(0x002, 0x01);
                                                    PHYSetLongRAMAddr(0x003, 0x08);
                                                    PHYSetLongRAMAddr(0x004, 0xC4);
                                                    PHYSetLongRAMAddr(0x005, 0xFF);
                                                    PHYSetLongRAMAddr(0x006, 0xFF);
                                                    PHYSetLongRAMAddr(0x007, 0xFF);
                                                    PHYSetLongRAMAddr(0x008, 0xFF);
                                                    PHYSetLongRAMAddr(0x009, 0xFF);  //Should be 0xFF for Sync Response
                                                    PHYSetLongRAMAddr(0x00A, 0xFF);	
                                                    PHYSetLongRAMAddr(0x00B, 0xFF);		
                                                    PHYSetLongRAMAddr(0x00C, 0xFF);		
                                                    PHYSetLongRAMAddr(0x00D, 0xFF);			
                                                    PHYSetLongRAMAddr(0x00E, 0xFF);			

                                                    PHYSetShortRAMAddr(WRITE_TXNCON,0b00000001);  //transmit packet without ACK requested 
													SendSyncRsp = FALSE;
												}
                                        if( PingPongBack )
                                        {
                                            BYTE j;
                                            
											
                                            PingPongBack = FALSE;
											Sync = FALSE;
											replay_count = 0;
                                            for(j = 0; j < PingPongPackage; j++)
                                            {
												if(ConsoleIsGetReady())
													goto ConsoleExit1;
												if(Received_EndPacket) {Send_Packet(END_RESPONSE); Received_EndPacket = FALSE;}
												if(!Sync)
												{
												 	j--;
													replay_count++;
													if(replay_count == 20) 
													{
													c("\rUnable to communicate with the other unit....");
													replay_count = 0;
													}
													//PrintDigit(j);
												}
												if(TX_LED_ON)
												{
													LED_1 = 1;
													TX_LED_ON = FALSE;
												}
												else
												{
													LED_1 = 0;
													TX_LED_ON = TRUE;
												}
  
                                                PHYSetLongRAMAddr(0x000,sizeof(PredefinedPacket));
                                                PHYSetLongRAMAddr(0x001,sizeof(PredefinedPacket));
                                                PHYSetLongRAMAddr(0x002, 0x01);
                                                PHYSetLongRAMAddr(0x003, 0x08);
                                                PHYSetLongRAMAddr(0x004, 0xC4);
                                                PHYSetLongRAMAddr(0x005, 0xFF);
                                                PHYSetLongRAMAddr(0x006, 0xFF);
                                                PHYSetLongRAMAddr(0x007, 0xFF);
                                                PHYSetLongRAMAddr(0x008, 0xFF);
                                                PHYSetLongRAMAddr(0x009, j+1);
                                                PHYSetLongRAMAddr(0x00A, Packageid);
                                                PHYSetLongRAMAddr(0x00B, AvgRSSI);
                                                PHYSetLongRAMAddr(0x00C, AvgLQI);
												i = rand();
                                                PHYSetLongRAMAddr(0x00D, i);
												i = rand();
                                                PHYSetLongRAMAddr(0x00E, i);

                                                PHYSetShortRAMAddr(WRITE_TXNCON,0b00000001);  //transmit packet without ACK requested 
                                                PingPongSend++;
                                                {
                                                    DWORD l;
                                                    WORD w;
                                                    
                                                    l = ((DWORD)packetDelay)*((DWORD)packetDelay);
                                        
                                                    do
                                                    {
                                                        for(w=0;w<2000;w++)
                                                        {
                                                        }
                                                    }
                                                    while(l--);
                                                } 
											
													
												                                                                                       
                                            }
                                            PrintDigit(j);
											c("\r\n");
											if(Packageid == 100) Packageid = 1;
											else Packageid++;
                                            while(!Received_EndRsp)
												{
													Send_Packet(END_PACKET);
												}
											Received_EndRsp = FALSE;
                                        } 
                                                if(Print_RXStats)
                                                {
													Send_Packet(END_RESPONSE);
													Print_RXStats = FALSE;
                                                    PingPongBack = TRUE;
                                                    PingPongTimeout = 0;
													if(display_count == 0)
													{
													c("\r\n");
													c("               Receiver                 |   Sender       \r\n");
                                                	c("Id    PacketCount     AvgRSSI     AvgLQI| PacketCount    \r\n");        												
													}
													display_count++;
													if(display_count == 20) display_count = 0;
													PrintDigit(Packageid);
													c("         ");
													AvgRSSI  = (BYTE)((RX_RSSI_VAL) / (packageGet));
													AvgLQI = (BYTE) ((RX_LQI_VAL) / (packageGet));
													PrintDigit(packageGet);
													c("         ");
													PrintChar(AvgRSSI);
													RSSI_dB = RSSIlookup(AvgRSSI);
													ConsolePutROMString((ROM char *)"(-");
													PrintDigit(RSSI_dB);
													ConsolePutROMString((ROM char *)"dBm)");
													c("   ");
													PrintChar(AvgLQI);
													c("        ");
													RX_RSSI_VAL = 0;
													RX_LQI_VAL = 0;
                                                    packageGet = 0;
                                                }
                                               
                                    }
                                    else
                                    {
ConsoleExit1:                           input = ConsoleInput();
                                        break;
                                    }
                                }
                                break;
                                
                            case 'b':
                            case 'B':
                                {
                                    BYTE i,j;
                                    if( packetDelay < 2 )
                                    {
                                        packetDelay = 2;
                                    }
                                    PingPongTest = TRUE;
                                    PingPongBack = TRUE;
                                    PingPongSend = 0;
                                    PingPongGet = 0;
                                    packageGet = 0;
									RX_RSSI_VAL = 0;
									RX_LQI_VAL = 0;
                                    PingPongTimeout = 0;
                                    PingPongStart = FALSE;
									Packageid = 0;

                                    c("\r\nPress Any Key to Return to the Main Menu.\r\n");
                                    while(1)
                                    {
                                        if(!ConsoleIsGetReady())
                                        {
											if(SendSyncRsp)
												{
													PHYSetLongRAMAddr(0x000,sizeof(PredefinedPacket));
                                                    PHYSetLongRAMAddr(0x001,sizeof(PredefinedPacket));
                                                    PHYSetLongRAMAddr(0x002, 0x01);
                                                    PHYSetLongRAMAddr(0x003, 0x08);
                                                    PHYSetLongRAMAddr(0x004, 0xC4);
                                                    PHYSetLongRAMAddr(0x005, 0xFF);
                                                    PHYSetLongRAMAddr(0x006, 0xFF);
                                                    PHYSetLongRAMAddr(0x007, 0xFF);
                                                    PHYSetLongRAMAddr(0x008, 0xFF);
                                                    PHYSetLongRAMAddr(0x009, 0xFF);  //Should be 0xFF for Sync Response
                                                    PHYSetLongRAMAddr(0x00A, 0xFF);	
                                                    PHYSetLongRAMAddr(0x00B, 0xFF);		
                                                    PHYSetLongRAMAddr(0x00C, 0xFF);		
                                                    PHYSetLongRAMAddr(0x00D, 0xFF);			
                                                    PHYSetLongRAMAddr(0x00E, 0xFF);			

                                                    PHYSetShortRAMAddr(WRITE_TXNCON,0b00000001);  //transmit packet without ACK requested 
													SendSyncRsp = FALSE;
												}
                                            if( PingPongBack )
                                            {
												if(Packageid  == 100) Packageid = 1;
												else Packageid++;
												Sync = FALSE;
												replay_count = 0;
                                                PingPongBack = FALSE;
                                                for(j = 0; j < PingPongPackage; j++)
                                                {
													if(Received_EndPacket) {Send_Packet(END_RESPONSE); Received_EndPacket = FALSE;}
													if(ConsoleIsGetReady())
													goto ConsoleExit2;
													if(!Sync) 
													{
													j--;  
													replay_count++;
													if(replay_count == 20)
													{
														replay_count = 0;
														c("\rUnable to communicate with the other unit");
													}
													//PrintDigit(j);
													}
                                                    PHYSetLongRAMAddr(0x000,sizeof(PredefinedPacket));
                                                    PHYSetLongRAMAddr(0x001,sizeof(PredefinedPacket));
                                                    PHYSetLongRAMAddr(0x002, 0x01);
                                                    PHYSetLongRAMAddr(0x003, 0x08);
                                                    PHYSetLongRAMAddr(0x004, 0xC4);
                                                    PHYSetLongRAMAddr(0x005, 0xFF);
                                                    PHYSetLongRAMAddr(0x006, 0xFF);
                                                    PHYSetLongRAMAddr(0x007, 0xFF);
                                                    PHYSetLongRAMAddr(0x008, 0xFF);
                                                    PHYSetLongRAMAddr(0x009, j+1);
                                                    PHYSetLongRAMAddr(0x00A, Packageid);	//insert the package id (1 - 100)
                                                    PHYSetLongRAMAddr(0x00B, AvgRSSI);		//avg RSSI
                                                    PHYSetLongRAMAddr(0x00C, AvgLQI);		//avg LQI
													i = rand();
                                                    PHYSetLongRAMAddr(0x00D, i);			//Random number 
													i = rand();
                                                    PHYSetLongRAMAddr(0x00E, i);			//Random number

                                                    PHYSetShortRAMAddr(WRITE_TXNCON,0b00000001);  //transmit packet without ACK requested 
                                                    PingPongSend++;
                                                    {
                                                        DWORD l;
                                                        WORD w;
                                                        
                                                        l = ((DWORD)packetDelay)*((DWORD)packetDelay);
                                            
                                                        do
                                                        {
                                                            for(w=0;w<2000;w++)
                                                            {
                                                            }
                                                        }
                                                        while(l--);
                                                    } 
													
                                                                                           
                                                }
												if(display_count == 0)
												{
												c("\r\n");
												c("      Sender       |            Receiver           \r\n");
                                                c("Id  PacketCount    |   PacketCount     AvgRSSI     AvgLQI\r\n");        												
												}
												display_count++;
												if(display_count == 20) display_count = 0;
												PrintDigit(Packageid);
												c("      ");
                                                PrintDigit(j);
												c("                 ");
	                                            while(!Received_EndRsp)
													{
														Send_Packet(END_PACKET);
													}
												Received_EndRsp = FALSE;
                                            } 
                                           
                                                    if(Print_RXStats)
                                                    {
                                                        PingPongBack = TRUE;
                                                        Send_Packet(END_RESPONSE);
														Print_RXStats = FALSE;
														AvgRSSI = (BYTE)((RX_RSSI_VAL)/(packageGet));
														AvgLQI = (BYTE)((RX_LQI_VAL) / (packageGet));
														PrintDigit(packageGet);
														c("        ");
														PrintChar(AvgRSSI);
														RSSI_dB = RSSIlookup(AvgRSSI);
														ConsolePutROMString((ROM char *)"(-");
														PrintDigit(RSSI_dB);
														ConsolePutROMString((ROM char *)"dBm)");
														c("    ");
														PrintChar(AvgLQI);
														c("\r\n");
												
														RX_RSSI_VAL = 0;
														RX_LQI_VAL = 0;
                                                        packageGet = 0;
                                                    }
                                           
                                              
                                        }
                                        else
                                        {
ConsoleExit2:                              input = ConsoleInput();
                                            break;
                                        }
                                    }
                                }
                                break;
                                
                            default:
                                goto PingPongTest_Input;
                        }     
                        
                        
                    }
                    break;
             
                
                
                case 'l':
                case 'L':ConsolePut(input);
                    {
                        BYTE i,data;
                        
                        c("\r\n-- Short RAM Addresses\r\nAddress\tValue\r\n");
                        for(i=0x00;i<=0x3F;i++)
                        {
                            data = PHYGetShortRAMAddr(i<<1);
                            c("Register 0x");
                            PrintChar(i);
                            c(":\t");
                            PrintChar(data);
                            c("\r\n");
                            if( (i & 0x0F) == 0x0F )
                            {
                                c("Press Any Key to Continue>>");
                                ConsoleInput();
                            }
                        }
                        
                        c("\r\n-- Long RAM Addresses\r\nAddress\tValue\r\n");
                        for(i=0x00;i<=0x4C;i++)
                        {
                            data = PHYGetLongRAMAddr(0x200+i);
                            c("Register 0x2");
                            PrintChar(i);
                            c(":\t");
                            PrintChar(data);
                            c("\r\n");
                            if( (i & 0x0F) == 0x0F )
                            {
                                c("Press Any Key to Continue>>");
                                ConsoleInput();
                            }
                           
                        }
                        c("\r\n");                        
                    }
                    goto MRF24J40_SubMenu;
                    break;
                
                default:Skip_submenu = TRUE;
                    	break;
    }
	if(Skip_submenu)
		{
		Skip_submenu = FALSE;
		goto MRF24J40_Wait;
		}
	else
    	goto MRF24J40_SubMenu;
	
}

/*********************************************************************
 * Function:        void BoardInit( void )
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    Board is initialized
 *
 * Overview:        This function configures the board for the PICDEM-z
 *                  MRF24J40 usage 
 *
 * Note:            This routine needs to be called before the function 
 *                  to initialize MiWi stack or any other function that
 *                  operates on the stack
 ********************************************************************/
#if defined(__18F4620) || defined(__18F87J11)
    void BoardInit(void)
    {
        #if defined(__18F4620)
            // Switches S2 and S3 are on RB5 and RB4 respectively. We want interrupt-on-change
            INTCON = 0x00;
        
            // There is no external pull-up resistors on S2 and S3. We will use internal pull-ups.
            // The MRF24J40 is using INT0 for interrupts
            // Enable PORTB internal pullups
            INTCON2 = 0x00;
            INTCON3 = 0x00;
        
            // Make PORTB as input - this is the RESET default
            TRISB = 0xff;
        
            // Set PORTC control signal direction and initial states
            // disable chip select
            LATC = 0xfd;
        
            // Set the SPI module for use by Stack
            TRISC = 0xD0;
        
            // Set the SPI module
            SSPSTAT = 0xC0;
            SSPCON1 = 0x20;
        
            // D1 and D2 are on RA0 and RA1 respectively, and CS of TC77 is on RA2.
            // Make PORTA as digital I/O.
            // The TC77 temp sensor CS is on RA2.
            ADCON1 = 0x0F;
        
            // Deselect TC77 (RA2)
            LATA = 0x04;
        
            // Make RA0, RA1, RA2 and RA4 as outputs.
            TRISA = 0xF8;
            
            PHY_CS = 1;             //deselect the MRF24J40
            PHY_CS_TRIS = 0;        //make chip select an output   
            
            RFIF = 0;               //clear the interrupt flag
            
            if(RB0 == 0)
            {
                RFIF = 1;
            }
            
            RFIE = 1;               //set interrupt enable flag
        
            IPEN = 1;
            
            GIEH = 1;
	
		#elif defined(__18F87J11)

		
//	 		LATC3               = 0;    // SCK
//    	    LATC5               = 1;    // SDO
        	TRISC3             = 0;    // SCK
        	TRISC4             = 1;    // SDI
        	TRISC5             = 0;    // SDO
			INTCON = 0x00;
			INTCON2 = 0x80;
			INTCON3 = 0x00;



		// Make PORTd as output - this drives LEDs
            
            TRISD = 0x00;
			LATD = 0x00;
			INTCON2bits.INTEDG0 = 0;
		//For REV1 MRF24J40MB daughter cards
			INTCON2bits.INTEDG1 = 0;
			INTCON3bits.INT1IP = 1;
		//End REV1
			PIE1bits.SSP1IE = 1;
			SSP1CON1 = 0x20;
			SSP1STAT = 0x40;
			
			PHY_RESETn = 0;
        	PHY_RESETn_TRIS = 0;
	        PHY_CS = 1;
    	    PHY_CS_TRIS = 0;
			PHY_WAKE_TRIS = 1;
            
            RFIF = 0;               //clear the interrupt flag
        
            RFIE = 1;
            
            IPEN = 1;
            INTCONbits.GIE = 1;            

        #else
        
            OSCCON = 0b01110110;
            OSCTUNE = 0b01000000;
            
            // Switches are on RB1 to RB4 respectively. We don't want interrupt-on-change
            INTCON = 0x00;
            
            // There is no external pull-up resistors on S2 and S3. We will use internal pull-ups.
            // The MRF24J40 is using INT0 for interrupts
            // Enable PORTB internal pullups
            INTCON2 = 0x80;
            INTCON3 = 0x00;
        
            // Make PORTB as input - this is the RESET default
            TRISB = 0xff;
        
            // Make PORTE as output - this drives LEDs
            LATE = 0x00;
            TRISE = 0x00;
        
            // Set PORTC control signal direction and initial states
            // disable chip select
            LATD = 0b11111011;
        
            // Set the SPI module for use by Stack
            TRISD = 0b00100000;
        
            // Set the SPI module
            SSP2STAT = 0xC0;
            SSP2CON1 = 0x20;
    
            TRISA = 0xFF;
            
            // D1 and D2 are on RA0 and RA1 respectively, and CS of TC77 is on RA2.
            // Make PORTA as digital I/O.
            // The TC77 temp sensor CS is on RA2.
            WDTCONbits.ADSHR = 1;
            ANCON0 = 0xDC;
            ANCON1 = 0xFF;
            WDTCONbits.ADSHR = 0;
            ADCON0 = 0x05;
            ADCON1 = 0xBA;
            
            PHY_CS = 1;             //deselect the MRF24J40
            PHY_CS_TRIS = 0;        //make chip select an output   
            
            RFIF = 0;               //clear the interrupt flag
        
            RFIE = 1;
            
            IPEN = 1;
            
            INTCON2bits.INTEDG0 = 0;
        
            GIEH = 1;
            
        #endif
    }
#else
    void BoardInit(void)
    {
        SPI1CON1 = 0b0000000100111110;
        SPI1STAT = 0x8000;
        
        AD1PCFGbits.PCFG2 = 1;
        
        LED_1_TRIS = 0;
        LED_2_TRIS = 0;
        
        INTCON2bits.INT1EP = 1;
        
        PHY_RESETn = 0;
        PHY_RESETn_TRIS = 0;
        PHY_CS = 1;
        PHY_CS_TRIS = 0;
        PHY_WAKE = 1;
        PHY_WAKE_TRIS = 0;
        RFIF = 0;
        RFIE = 1;
    }
#endif    

/*********************************************************************
 * Function:        void MRF24J40Init(void)
 *
 * PreCondition:    BoardInit (or other initialzation code is required)
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    MRF24J40 is initialized
 *
 * Overview:        This function initializes the MRF24J40 and is required
 *                  before stack operation is available
 ********************************************************************/
void MRF24J40Init(void)
{  
    BYTE i;
    WORD j;
	

MRF24J40_MRFInit:
    PHY_RESETn = 0;
	//reset the MRF24J40 by pulling the RESET pin13 low
    for(j=0;j<(WORD)300;j++){}
    PHY_RESETn = 1;
    for(j=0;j<(WORD)300;j++){}
  
    /* do a soft reset */
    PHYSetShortRAMAddr(WRITE_SOFTRST,0x07);
    do
    {
        i = PHYGetShortRAMAddr(READ_SOFTRST);
    }
    while((i&0x07) != (BYTE)0x00);   

	i = PHYGetShortRAMAddr(READ_ORDER);
	if(i!= 0xFF)
		{
		ConsolePutROMString((ROM char *)"\r\nMRF24J40 Daughter card not inserted");
		ConsolePutROMString((ROM char *)"\r\nInsert Daughter card and Press (Ctrl + X) to Initialize MRF24J40");
		c("\r\n>>");
			while(1)
				{
				if(ConsoleIsGetReady())
					{
						input = ConsoleGet();
						if(input == 24)
							{
							goto MRF24J40_MRFInit;
							}
					}
				}	 

		}

  	PHYSetShortRAMAddr(WRITE_PACON2,0x98);
				//PACON2 = 0x98, Initialize FFOEN=1 and TXONTS = 0x6
	PHYSetShortRAMAddr(WRITE_TXSTBL,0x95);
				//Initialize RFSTBL = 0x9
	PHYSetLongRAMAddr(RFCON1,0x01);
				//Initialize VCOOPT=1
	PHYSetLongRAMAddr(RFCON2,0x80);
				//Enable PLL
	PHYSetLongRAMAddr(RFCON6,0x90);
				//Initialize TXFIL=1, 20MRECVR=1
	PHYSetLongRAMAddr(RFCON7, 0x80);
				//Initialize SLPCLKSEL = 0x2 (100KHz internal oscialltor)
	PHYSetLongRAMAddr(RFCON8, 0x10);
				//Initialize RFVCO =1
	PHYSetLongRAMAddr(SLPCON1, 0x21);
				//Initialize CLKOUTEN=1 and SLPCLKDIV = 0x01
	PHYSetShortRAMAddr(WRITE_BBREG2,0x80);
				//Set CCA mode to ED
	PHYSetShortRAMAddr(WRITE_CCAEDTH,0x60);
				//Set CCA-ED Threshold
	PHYSetShortRAMAddr(WRITE_BBREG6,0x40);
				//Set appended RSSI value to RX FIFO
    PHYSetShortRAMAddr(WRITE_INTCON,0xF6);
		//INTCON (0x32) = 0xF6 - Enables only TXNIF and RXIF interrupts	
    SetChannel(CHANNEL_11);
		//set operating channel as channel 11
	for(j=0;j<(WORD)800;j++)
	{
		Nop();
	}
		//Delay for 192 us after RF State Machine Reset
    do
    {
        i = PHYGetLongRAMAddr(RFSTATE);
    }
    while((i&0xA0) != 0xA0);  
		//Wait until the RFSTATE machine indicates RX state

    /* Program the short MAC Address, 0xffff */
    PHYSetShortRAMAddr(WRITE_SADRL,0xFF);
    PHYSetShortRAMAddr(WRITE_SADRH,0xFF);
			//load the short address of the device with 0xffff which means that it will be ignored upon receipt
    PHYSetShortRAMAddr(WRITE_PANIDL,0xFF);
    PHYSetShortRAMAddr(WRITE_PANIDH,0xFF);
			//load the pan address also with 0xffff; 
    
    /* Program Long MAC Address*/
    for(i=0;i<(BYTE)8;i++)
    {
        PHYSetShortRAMAddr(WRITE_EADR0+i*2,myLongAddress[i]);
    }
			//load the long address as 07,06,..    

		
		PHYSetShortRAMAddr(WRITE_RXMCR, 0x21);
							// Automatic acknowledgement is not transmitted for received packets by TXMAC
							//ERRpkt mode bit is set to 0
							//Promiscuous packet mode bit is set to 1
			
}


/*********************************************************************
 * Function:        void SetChannel(BYTE channel)
 *
 * PreCondition:    MRF24J40 is initialized
 *
 * Input:           BYTE channel - this is the channel that you wish
 *                  to operate on.  This should be CHANNEL_11, CHANNEL_12,
 *                  ..., CHANNEL_26. 
 *
 * Output:          None
 *
 * Side Effects:    the MRF24J40 now operates on that channel
 *
 * Overview:        This function sets the current operating channel
 *                  of the MRF24J40
 ********************************************************************/
void SetChannel(BYTE channel)
{
    
    PHYSetLongRAMAddr(RFCON0, (channel | 0x02));
				//RFCON0 = 0x02 for Channel number 11
    PHYSetShortRAMAddr(WRITE_RFCTL,0x04);
    PHYSetShortRAMAddr(WRITE_RFCTL,0x00);
				//Reset RF State machine
				//no delay loop is present after performing RF State machine reset
}

/*********************************************************************
 * Function:        void PHYSetLongRAMAddr(WORD address, BYTE value)
 *
 * PreCondition:    Communication port to the MRF24J40 initialized
 *
 * Input:           WORD address is the address of the LONG RAM address
 *                      that you want to write to
 *                  BYTE value is the value that you want to write to
 *                      that register
 *
 * Output:          None
 *
 * Side Effects:    The register value is changed
 *
 * Overview:        This function writes a value to a LONG RAM address
 ********************************************************************/
void PHYSetLongRAMAddr(WORD address, BYTE value)
{
    volatile BYTE tmpRFIE = RFIE;
    RFIE = 0;
		//disable interrupts to the microcontroller temporarily
    PHY_CS = 0;
		//pull the chip select low (idle low)
    SPIPut((((BYTE)(address>>3))&0b01111111)|0x80);
    SPIPut((((BYTE)(address<<5))&0b11100000)|0x10);
		//12 bit address 1 A9.....A0 0/1
		//converting to 16 bit by setting last nibble to 0s
    SPIPut(value);
		//8 bit value
    PHY_CS = 1;
		//pull the chip select high - not selecting the SPI slave
    RFIE = tmpRFIE;
		//return to previous interrupt status
}

/*********************************************************************
 * Function:        void PHYSetShortRAMAddr(BYTE address, BYTE value)
 *
 * PreCondition:    Communication port to the MRF24J40 initialized
 *
 * Input:           BYTE address is the address of the short RAM address
 *                      that you want to write to.  Should use the
 *                      WRITE_ThisAddress definition in the MRF24J40 
 *                      include file.
 *                  BYTE value is the value that you want to write to
 *                      that register
 *
 * Output:          None
 *
 * Side Effects:    The register value is changed
 *
 * Overview:        This function writes a value to a short RAM address
 ********************************************************************/
void PHYSetShortRAMAddr(BYTE address, BYTE value)
{
    volatile BYTE tmpRFIE = RFIE;
    RFIE = 0;
		//disabling the interrupt to the microcontroller
    PHY_CS = 0;     
		//Pull the Chip select signal to low  
    SPIPut(address);
		//8 bit address (short address 0, 6 bit address, read/write (0/1))
    SPIPut(value);	
		//8 bit value
    PHY_CS = 1;
		//disable chip select
    RFIE = tmpRFIE;
		//Interrupt status is set back to previous state
}


/*********************************************************************
 * Function:        void PHYGetShortRAMAddr(BYTE address, BYTE value)
 *
 * PreCondition:    Communication port to the MRF24J40 initialized
 *
 * Input:           BYTE address is the address of the short RAM address
 *                      that you want to read from.  Should use the
 *                      READ_ThisAddress definition in the MRF24J40 
 *                      include file.
 *
 * Output:          BYTE the value read from the specified short register
 *
 * Side Effects:    None
 *
 * Overview:        This function reads a value from a short RAM address
 ********************************************************************/
BYTE PHYGetShortRAMAddr(BYTE address)
{
    BYTE toReturn;
    volatile BYTE tmpRFIE = RFIE;
    RFIE = 0;
		//disable interrupts to the microcontroller
    PHY_CS = 0;
		//select the slave device
    SPIPut(address);
		//put the address value on SPI port
    toReturn = SPIGet();
		//Read back the data
    PHY_CS = 1;
		//deselect the SPI slave device
    RFIE = tmpRFIE;
		//return to previous interrupt status
    return toReturn;
		//return the read value to the calling function
}


/*********************************************************************
 * Function:        BYTE PHYGetLongRAMAddr(WORD address)
 *
 * PreCondition:    Communication port to the MRF24J40 initialized
 *
 * Input:           WORD address is the address of the long RAM address
 *                      that you want to read from.  
 *
 * Output:          BYTE the value read from the specified Long register
 *
 * Side Effects:    None
 *
 * Overview:        This function reads a value from a long RAM address
 ********************************************************************/
BYTE PHYGetLongRAMAddr(WORD address)
{
    BYTE toReturn;
    volatile BYTE tmpRFIE = RFIE;
    RFIE = 0;
		//temporarily disable interrupts to the microcontroller
    PHY_CS = 0;
		//select the slave device for communication
    SPIPut(((address>>3)&0b01111111)|0x80);
    SPIPut(((address<<5)&0b11100000));
		//put the 12 bit long address on the SDO line
    toReturn = SPIGet();
		//Read back the value from SPI port
    PHY_CS = 1;
		//Deselect the Slave device
    RFIE = tmpRFIE;
		//restore the previous interrupt status
    return toReturn;	
		//return the value read to the calling function
}


/*********************************************************************
 * Function:        void PHYSetDeviceAddress(WORD PANID, WORD shortAddress)
 *
 * PreCondition:    Communication port to the MRF24J40 initialized
 *
 * Input:           WORD PANID is the PANID you want to operate on
 *                  WORD shortAddress is the short address you want to use on that network
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        This function sets the short address for the IEEE address
 *                      filtering
 ********************************************************************/
void PHYSetDeviceAddress(WORD PANID, WORD shortAddress)
{
    PHYSetShortRAMAddr(WRITE_SADRL,(BYTE)(PANID&0x00FF));
    PHYSetShortRAMAddr(WRITE_SADRH,(BYTE)(PANID>>8));
		//set the short address of the device to value PANID
    PHYSetShortRAMAddr(WRITE_PANIDL,(BYTE)(shortAddress & 0x00FF));
    PHYSetShortRAMAddr(WRITE_PANIDH,(BYTE)(shortAddress>>8));
		//set the PANID to the value the value shortaddress
}

/*********************************************************************
 * Function:        void Send_Packet(BYTE Packet_Type)
 *
 * PreCondition:    None
 *
 * Input:           Packet Type
 *                  
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        This function transmits packet defined by "Packet_Type"
  ********************************************************************/
void Send_Packet(BYTE Packet_Type)
{
 PHYSetLongRAMAddr(0x000,sizeof(PredefinedPacket));
 PHYSetLongRAMAddr(0x001,sizeof(PredefinedPacket));
 PHYSetLongRAMAddr(0x002, 0x01);
 PHYSetLongRAMAddr(0x003, 0x08);
 PHYSetLongRAMAddr(0x004, 0xC4);
 PHYSetLongRAMAddr(0x005, 0xFF);
 PHYSetLongRAMAddr(0x006, 0xFF);
 PHYSetLongRAMAddr(0x007, 0xFF);
 PHYSetLongRAMAddr(0x008, 0xFF);
 PHYSetLongRAMAddr(0x009, Packet_Type);
 PHYSetLongRAMAddr(0x00A, 0xFF);	
 PHYSetLongRAMAddr(0x00B, 0xFF);		
 PHYSetLongRAMAddr(0x00C, 0xFF);		
 PHYSetLongRAMAddr(0x00D, 0xFF);			
 PHYSetLongRAMAddr(0x00E, 0xFF);			

 PHYSetShortRAMAddr(WRITE_TXNCON,0b00000001);  //transmit packet without ACK requested 
 {
     DWORD l;
     WORD w;
     
     l = ((DWORD)packetDelay)*((DWORD)packetDelay);

     do
     {
         for(w=0;w<2000;w++)
         {
         }
     }
     while(l--);
 } 

}

/*********************************************************************
 * Function:        char RSSIlookup(BYTE RSSIval)
 *
 * PreCondition:    None
 *
 * Input:           BYTE RSSI value that you want to convert
 *
 * Output:          RSSI value in dB - decibels
 *
 * Side Effects:    None
 *
 * Overview:        This function converts the integer RSSI value in to
 *                      RSSI value in dB
 ********************************************************************/

BYTE RSSIlookup(BYTE RSSIval)
{
return RSSIlookupTable[RSSIval];
}

/*********************************************************************
 * Function:        void HighISR(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    Various flags and registers set.
 *
 * Overview:        This is the interrupt handler for the MRF24J40 and
 *                  MiWi stack.
 ********************************************************************/
#if defined(__18F4620) || defined(__18F87J11)
    #pragma interruptlow HighISR
    void HighISR(void)
#else
    void _ISRFAST __attribute__((interrupt, auto_psv)) _INT1Interrupt(void)
#endif
{       
    if(RFIE && RFIF)
    {                    
        								//clear the interrupt flag as soon as possible such that another interrupt can
        								//occur quickly.
        RFIF = 0;
        
        								//create a new scope for the MRF24J40 interrupts so that we can clear the interrupt
        								//flag quickly and then handle the interrupt that we have already received
        {
            MRF24J40_IFREG flags;        

            							//read the interrupt status register to see what caused the interrupt        
            flags.Val=PHYGetShortRAMAddr(READ_INTSTAT);
            							//PrintChar(flags.Val);
           
            if(flags.bits.RF_TXIF)
            {
                						//if the TX interrupt was triggered
                BYTE_VAL results;
               
                						//read out the results of the transmission
                results.Val = PHYGetShortRAMAddr(READ_TXSTAT);
                
                if(results.bits.b0 == 1)
                {
                    					//the transmission wasn't successful and the number
                    					//of retries is located in bits 7-6 of TXSTAT
                    					//failed to Transmit
                }
                else
                {
                    					//transmission successful
                    					//MAC ACK received if requested else packet passed CCA
                }

            }
            
            if(flags.bits.RF_RXIF)
            {   
                volatile BYTE i,j,RSSI_VAL, LQI_VAL;
                   
                						//packet received
                						//Need to read it out
                
                i=PHYGetLongRAMAddr(0x300); //first byte of the receive buffer is the packet length (this 
                j=PHYGetLongRAMAddr(0x308);
                numPacketsRx.Val++;

                RSSI_VAL = PHYGetLongRAMAddr(0x300+i+2);
                LQI_VAL = PHYGetLongRAMAddr(0x300+i+1);
				Testid = PHYGetLongRAMAddr(0x309);
                if( PingPongTest )
                {
                    if(RX_LED_ON)
						{
						LED_2 = 1;
						RX_LED_ON = FALSE;
						}
					else
						{
						LED_2 = 0;
						RX_LED_ON = TRUE;
						}
						
					PingPongGet++;
					
					if((j!= 0x00) & (j!= 0xFF) & (j!=0xFD) & (j!=0xFE))
					{
					RX_RSSI_VAL = RX_RSSI_VAL + RSSI_VAL;
					RX_LQI_VAL = RX_LQI_VAL + LQI_VAL;
                    if(Testid == Packageid) packageGet++;
					PingPongStart = FALSE;
					}
					
                    
                    if(j == 0x00) SendSyncRsp = TRUE;
					if(j == 0xFF) Sync = TRUE;
					if(j == 0xFD) Print_RXStats = TRUE;
					if(j == 0xFE) Received_EndRsp = TRUE;
                    PHYSetShortRAMAddr(WRITE_RXFLUSH, 0x01);
                    
                    return;
                    
                }
                else if(j == 0xAA )
                {
                    StartSendingPER = 1; 
                }
                else
                {
                    if( ReceivingMode )
                    {
						PacketReceived = TRUE;
                    }
					 if( RxPacketCountMode )
						{
							RxPacket_Count++;
							RxPacket_RSSI = RSSI_VAL;
						}
                }
                

                PHYSetShortRAMAddr(WRITE_RXFLUSH, 0x01);
            } 															//end of RXIF check
        } 																//end of scope of RF interrupt handler
    } 																	//end of if(RFIE && RFIF)
    
} 																		//end of interrupt handler

//******************************************************************************
// Interrupt Vectors
//******************************************************************************
#if defined(__18F4620) || defined(__18F87J11)
#pragma code highVector=0x08
void HighVector (void)
{
    _asm goto HighISR _endasm
}
#pragma code /* return to default code section */
#endif

#if defined(__18F4620) || defined(__18F87J11)
#pragma code lowhVector=0x18
void LowVector (void)
{
    _asm goto HighISR _endasm
}
#pragma code /* return to default code section */
#endif
